use serde::Deserialize;
use uuid::Uuid;

use crate::common::FromResponse;
use crate::types::{RelationshipType, ReportCategory};
use crate::v5::schema::LocalizedString;

/// Report reason response object.
#[derive(Clone, Debug, Deserialize, PartialEq)]
#[serde(rename_all = "camelCase")]
pub struct ReportReasonAttributes {
    pub reason: LocalizedString,
    pub details_required: bool,
    pub category: ReportCategory,
    pub version: u32,
}

#[derive(Clone, Debug, Deserialize, PartialEq)]
#[serde(rename_all = "camelCase")]
pub struct ReportReasonObject<A, T = RelationshipType> {
    pub id: Uuid,
    pub type_: T,
    pub attributes: A,
}

// This is a workaround to the different structure MangaDex returns for the results.
// Each reason does *not* come with the typical `result`, `data`, and `relationships` fields parenting the `ApiObject`.
//
// Instead of:
//
// ```json
// {
//     "results": [
//         {
//             "result": "ok",
//             "data": {
//                 "id": "497f6eca-6276-4993-bfeb-53cbbbba6f08",
//                 "type": "report_reason",
//                 "attributes": {
//                     // ...
//                 },
//                 "relationships": []
//             }
//         }
//     ],
//     "limit": 10,
//     "offset": 0,
//     "total": 10
// }
//
// MangaDex returns this different JSON structure:
//
// ```json
// {
//     "results": [
//         {
//             // There is no `result`, `data`, or `relationships` fields in the response body.
//             "id": "497f6eca-6276-4993-bfeb-53cbbbba6f08",
//             "type": "report_reason",
//             "attributes": {
//                 // ...
//             }
//         }
//     ],
//     "limit": 10,
//     "offset": 0,
//     "total": 10
// }
#[derive(Clone, Debug, Deserialize, PartialEq)]
pub struct ReportReasonListResponse {
    pub results: Vec<ReportReasonObject<ReportReasonAttributes>>,
    pub limit: u32,
    pub offset: u32,
    pub total: u32,
}

impl FromResponse for ReportReasonListResponse {
    type Response = Self;

    fn from_response(value: Self::Response) -> Self {
        value
    }
}
