//! General CustomList information.

use serde::Deserialize;
use uuid::Uuid;

use crate::common::FromResponse;
use crate::types::{CustomListVisibility, RelationshipType};

#[derive(Clone, Debug, Deserialize, PartialEq)]
#[serde(rename_all = "camelCase")]
pub struct CustomListAttributes {
    pub name: String,
    pub visibility: CustomListVisibility,
    pub version: u32,
}

#[derive(Clone, Debug, Deserialize, PartialEq)]
#[serde(rename_all = "camelCase")]
pub struct CustomListObject<A, T = RelationshipType> {
    pub id: Uuid,
    pub type_: T,
    pub attributes: A,
}

// This is a workaround to the different structure MangaDex returns for the results.
// Each reason does *not* come with the typical `result`, `data`, and `relationships` fields parenting the `ApiObject`.
//
// Instead of:
//
// ```json
// {
//     "results": [
//         {
//             "result": "ok",
//             "data": {
//                 "id": "497f6eca-6276-4993-bfeb-53cbbbba6f08",
//                 "type": "custom_list",
//                 "attributes": {
//                     // ...
//                 },
//                 "relationships": []
//             },
//         }
//     ],
//     "limit": 10,
//     "offset": 0,
//     "total": 10
// }
//
// MangaDex returns this different JSON structure:
//
// ```json
// {
//     "results": [
//         {
//             // There is no `result`, `data`, or `relationships` fields in the response body.
//             "id": "497f6eca-6276-4993-bfeb-53cbbbba6f08",
//             "type": "custom_list",
//             "attributes": {
//                 // ...
//             }
//         }
//     ],
//     "limit": 10,
//     "offset": 0,
//     "total": 10
// }
#[derive(Clone, Debug, Deserialize, PartialEq)]
pub struct CustomListsListResponse {
    pub results: Vec<CustomListObject<CustomListAttributes>>,
    pub limit: u32,
    pub offset: u32,
    pub total: u32,
}

impl FromResponse for CustomListsListResponse {
    type Response = Self;

    fn from_response(value: Self::Response) -> Self {
        value
    }
}
