//! Builder for the scanlation group view endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-group-id>
//!
//! # Examples
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let group_id = Uuid::new_v4();
//! let group_res = client
//!     .scanlation_group()
//!     .view()
//!     .group_id(&group_id)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("group view: {:?}", group_res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::types::RelationshipType;
use crate::v5::schema::GroupResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned")]
pub struct GetGroup<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[serde(skip)]
    pub group_id: &'a Uuid,

    #[builder(setter(each = "add_relationship"), default)]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET ("/group/{}", group_id),
    #[query] GetGroup<'_>,
    #[flatten_result] GroupResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path_regex};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn get_group_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let group_id = Uuid::new_v4();
        let date_time: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "result": "ok",
            "data": {
                "id": group_id,
                "type": "scanlation_group",
                "attributes": {
                    "name": "Scanlation Group",
                    "website": "https://example.org",
                    "ircServer": null,
                    "ircChannel": null,
                    "discord": null,
                    "contactEmail": null,
                    "description": null,
                    "locked": false,
                    "official": false,
                    "verified": false,
                    "version": 1,
                    "createdAt": date_time,
                    "updatedAt": date_time
                },
                "relationships": []
            }
        });

        Mock::given(method("GET"))
            .and(path_regex(r"/group/[0-9a-fA-F-]+"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let _ = mangadex_client
            .scanlation_group()
            .get()
            .group_id(&group_id)
            .build()?
            .send()
            .await?;

        Ok(())
    }
}
