//! Manga endpoint handler.
//!
//! <https://api.mangadex.org/docs.html#tag/Manga>

pub(crate) mod add_to_custom_list;
mod aggregate;
mod create;
mod delete;
mod feed;
mod follow;
mod get;
mod get_manga_read_chapters;
mod get_read_chapters;
pub(crate) mod list;
mod list_tags;
mod random;
mod reading_status;
mod reading_statuses;
pub(crate) mod remove_from_custom_list;
mod unfollow;
mod update;
mod update_reading_status;

use crate::v5::manga::add_to_custom_list::AddMangaToCustomListBuilder;
use crate::v5::manga::aggregate::GetMangaAggregateBuilder;
use crate::v5::manga::create::CreateMangaBuilder;
use crate::v5::manga::delete::DeleteMangaBuilder;
use crate::v5::manga::feed::GetMangaFeedBuilder;
use crate::v5::manga::follow::FollowMangaBuilder;
use crate::v5::manga::get::GetMangaBuilder;
use crate::v5::manga::get_manga_read_chapters::GetMangaReadChaptersBuilder;
use crate::v5::manga::get_read_chapters::GetReadChaptersBuilder;
use crate::v5::manga::list::ListMangaBuilder;
use crate::v5::manga::list_tags::ListTagsBuilder;
use crate::v5::manga::random::GetRandomMangaBuilder;
use crate::v5::manga::reading_status::MangaReadingStatusBuilder;
use crate::v5::manga::reading_statuses::MangaReadingStatusesBuilder;
use crate::v5::manga::remove_from_custom_list::RemoveMangaFromCustomListBuilder;
use crate::v5::manga::unfollow::UnfollowMangaBuilder;
use crate::v5::manga::update::UpdateMangaBuilder;
use crate::v5::manga::update_reading_status::UpdateMangaReadingStatusBuilder;
use crate::v5::user::followed_manga_feed::GetFollowedMangaFeedBuilder;
use crate::HttpClientRef;

/// Manga endpoint handler.
#[derive(Debug)]
pub struct MangaBuilder {
    http_client: HttpClientRef,
}

impl MangaBuilder {
    pub fn new(http_client: HttpClientRef) -> Self {
        Self { http_client }
    }

    /// Search a list of Manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-search-manga>
    pub fn list(&self) -> ListMangaBuilder {
        ListMangaBuilder::default().http_client(self.http_client.clone())
    }

    /// Search a list of Manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-search-manga>
    ///
    /// This is an alias for the [`Self::list()`] function.
    pub fn search(&self) -> ListMangaBuilder {
        self.list()
    }

    /// Get Manga volumes and chapters.
    ///
    /// <https://api.mangadex.org/docs.html#tag/Manga/paths/~1manga~1{id}~1aggregate/get>
    pub fn aggregate(&self) -> GetMangaAggregateBuilder {
        GetMangaAggregateBuilder::default().http_client(self.http_client.clone())
    }

    /// View a single Manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-id>
    pub fn get(&self) -> GetMangaBuilder {
        GetMangaBuilder::default().http_client(self.http_client.clone())
    }

    /// View a single Manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-id>
    ///
    /// This is an alias for [`Self::get()`] to maintain backwards-compatibility.
    pub fn view(&self) -> GetMangaBuilder {
        self.get()
    }

    /// Update a manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/put-manga-id>
    pub fn update(&self) -> UpdateMangaBuilder {
        UpdateMangaBuilder::default().http_client(self.http_client.clone())
    }

    /// Delete a manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/delete-manga-id>
    pub fn delete(&self) -> DeleteMangaBuilder {
        DeleteMangaBuilder::default().http_client(self.http_client.clone())
    }

    /// Add manga to a custom list.
    ///
    /// <https://api.mangadex.org/docs.html#operation/post-manga-id-list-listId>
    pub fn add_to_custom_list(&self) -> AddMangaToCustomListBuilder {
        AddMangaToCustomListBuilder::default().http_client(self.http_client.clone())
    }

    /// Remove manga from a custom list.
    ///
    /// <https://api.mangadex.org/docs.html#operation/delete-manga-id-list-listId>
    pub fn remove_from_custom_list(&self) -> RemoveMangaFromCustomListBuilder {
        RemoveMangaFromCustomListBuilder::default().http_client(self.http_client.clone())
    }

    /// Get the manga feed (chapter list) of manga the logged-in user follows.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-user-follows-manga-feed>
    ///
    /// Alias to [`MangaDexClient::user().followed_manga_feed()`](crate::v5::user::followed_manga_feed).
    pub fn followed_manga_feed(&self) -> GetFollowedMangaFeedBuilder {
        GetFollowedMangaFeedBuilder::default().http_client(self.http_client.clone())
    }

    /// Follow a manga for the logged-in user.
    ///
    /// <https://api.mangadex.org/docs.html#operation/post-manga-id-follow>
    pub fn follow(&self) -> FollowMangaBuilder {
        FollowMangaBuilder::default().http_client(self.http_client.clone())
    }

    /// Unfollow a manga for the logged-in user.
    ///
    /// <https://api.mangadex.org/docs.html#operation/delete-manga-id-follow>
    pub fn unfollow(&self) -> UnfollowMangaBuilder {
        UnfollowMangaBuilder::default().http_client(self.http_client.clone())
    }

    /// Get recent chapters for a Manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-id-feed>
    pub fn feed(&self) -> GetMangaFeedBuilder {
        GetMangaFeedBuilder::default().http_client(self.http_client.clone())
    }

    /// Get a random manga, chosen by MangaDex.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-random>
    pub fn random(&self) -> GetRandomMangaBuilder {
        GetRandomMangaBuilder::default().http_client(self.http_client.clone())
    }

    /// Create a new manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/post-manga>
    pub fn create(&self) -> CreateMangaBuilder {
        CreateMangaBuilder::default().http_client(self.http_client.clone())
    }

    /// Get a list of chapter IDs that are marked as read for the specified manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-chapter-readmarkers>
    pub fn get_manga_read_chapters(&self) -> GetMangaReadChaptersBuilder {
        GetMangaReadChaptersBuilder::default().http_client(self.http_client.clone())
    }

    /// Get a list of chapter IDs that are marked as read for the given manga IDs.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-chapter-readmarkers-2>
    pub fn get_read_chapters(&self) -> GetReadChaptersBuilder {
        GetReadChaptersBuilder::default().http_client(self.http_client.clone())
    }

    /// List all of the available tags.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-tag>
    pub fn list_tags(&self) -> ListTagsBuilder {
        ListTagsBuilder::default().http_client(self.http_client.clone())
    }

    /// Get the reading status for a given followed manga.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-id-status>
    pub fn reading_status(&self) -> MangaReadingStatusBuilder {
        MangaReadingStatusBuilder::default().http_client(self.http_client.clone())
    }

    /// Get the reading statuses for all followed manga for the logged-in user.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-manga-status>
    pub fn reading_statuses(&self) -> MangaReadingStatusesBuilder {
        MangaReadingStatusesBuilder::default().http_client(self.http_client.clone())
    }

    /// Update the reading status for a manga.
    ///
    /// Using a `None` (`null`) value in the `status` field will remove the reading status.
    ///
    /// <https://api.mangadex.org/docs.html#operation/post-manga-id-status>
    pub fn update_reading_status(&self) -> UpdateMangaReadingStatusBuilder {
        UpdateMangaReadingStatusBuilder::default().http_client(self.http_client.clone())
    }
}
