//! Builder for the account-creation endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/post-account-create>
//!
//! # Examples
//!
//! ```rust
//! use mangadex_api::v5::{MangaDexClient, Username, Password};
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let account_create_res = client
//!     .account()
//!     .create()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter2")?)
//!     .email("test@example.com")
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("account create: {:?}", account_create_res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;

use crate::v5::schema::UserResponse;
use crate::v5::{HttpClientRef, Password, Username};

/// Create a new account.
///
/// Makes a request to `POST /account/create`.
#[derive(Debug, Builder, Serialize, Clone)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option))]
pub struct CreateAccount<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    /// Unique username, used for logging in.
    ///
    /// Min: 1 character
    ///
    /// Max: 64 characters
    pub username: Username,

    /// Set the account's password.
    ///
    /// Min: 8 characters
    ///
    /// Max: 1024 characters
    pub password: Password,

    /// E-mail address; used for notifications and account recovery from MangaDex.
    pub email: &'a str,
}

endpoint! {
    POST "/account/create",
    #[body] CreateAccount<'_>,
    #[flatten_result] UserResponse
}

#[cfg(test)]
mod tests {
    use fake::faker::internet::en::{Password, SafeEmail};
    use fake::faker::lorem::en::Word;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use wiremock::matchers::{body_json, header, method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::v5::{Password as MDPassword, Username};
    use crate::{HttpClient, MangaDexClient};
    use uuid::Uuid;

    #[tokio::test]
    #[ignore = "Can't get JSON bodies don't match"]
    async fn create_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let username: String = Word().fake();
        let password: String = Password(8..1024).fake();
        let email: String = SafeEmail().fake();

        let expected_body = json!({
            "username": username,
            "password": password,
            "email": email
        });
        let response_body = json!({
            "result": "ok",
            "data": {
                "id": Uuid::new_v4(),
                "type": "user",
                "attributes": {
                    "username": username,
                    "roles": [
                        "ROLE_MEMBER",
                        "ROLE_GROUP_MEMBER",
                        "ROLE_GROUP_LEADER"
                    ],
                    "version": 1
                },
                "relationships": [
                    {
                        "id": "a3219a4f-73c0-4213-8730-05985130539a",
                        "type": "scanlation_group"
                    }
                ]
            },
            "relationships": []
        });

        Mock::given(method("POST"))
            .and(path("/account/create"))
            .and(header("Content-Type", "application/json"))
            .and(body_json(expected_body))
            .respond_with(ResponseTemplate::new(201).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let _ = mangadex_client
            .account()
            .create()
            .username(Username::parse(&username)?)
            .password(MDPassword::parse(&password)?)
            .email(email.as_str())
            .build()?
            .send()
            .await?;

        Ok(())
    }
}
