//! Builder for the user list endpoint.
//!
//! Authentication is required. This can be done by logging in.
//!
//! <https://api.mangadex.org/docs.html#operation/get-user>
//!
//! # Examples
//!
//! ```rust
//! use mangadex_api::types::MangaStatus;
//! use mangadex_api::v5::{MangaDexClient, Username, Password};
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let _login_res = client
//!     .auth()
//!     .login()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter23")?)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let users_res = client
//!     .user()
//!     .list()
//!     .username("holo")
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("users: {:?}", users_res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::types::UserSortOrder;
use crate::v5::schema::UserListResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder, Default)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), default, pattern = "owned")]
pub struct ListUser {
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub limit: Option<u32>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub offset: Option<u32>,
    #[builder(setter(each = "add_user_id"))]
    #[serde(rename = "ids")]
    pub user_ids: Vec<Uuid>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub username: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub order: Option<UserSortOrder>,
}

endpoint! {
    GET "/user",
    #[query auth] ListUser,
    UserListResponse
}

#[cfg(test)]
mod tests {
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{header, method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::v5::AuthTokens;
    use crate::{Error, HttpClient, MangaDexClient};

    #[tokio::test]
    async fn list_user_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let user_id = Uuid::new_v4();
        let response_body = json!({
            "results": [
                {
                    "result": "ok",
                    "data": {
                        "id": user_id,
                        "type": "user",
                        "attributes": {
                            "username": "myusername",
                            "roles": [
                                "ROLE_MEMBER",
                                "ROLE_GROUP_MEMBER",
                                "ROLE_GROUP_LEADER"
                            ],
                            "version": 1
                        },
                        "relationships": [
                            {
                                "id": "a3219a4f-73c0-4213-8730-05985130539a",
                                "type": "scanlation_group"
                            }
                        ]
                    },
                    "relationships": []
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        });

        Mock::given(method("GET"))
            .and(path("/user"))
            .and(header("Authorization", "Bearer sessiontoken"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .user()
            .search()
            .limit(1u32)
            .build()?
            .send()
            .await?;

        let group = res.results[0].as_ref().unwrap();
        assert_eq!(group.data.id, user_id);
        assert_eq!(group.data.attributes.username, "myusername");
        assert_eq!(group.data.attributes.version, 1);

        Ok(())
    }

    #[tokio::test]
    #[ignore = "Handling list endpoint errors needs work/refactoring"]
    async fn list_manga_handles_400() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let error_id = Uuid::new_v4();

        let response_body = json!({
            "result": "error",
            "errors": [{
                "id": error_id.to_string(),
                "status": 400,
                "title": "Invalid limit",
                "detail": "Limit must be between 1 and 100"
            }]
        });

        Mock::given(method("GET"))
            .and(path("/user"))
            .respond_with(ResponseTemplate::new(400).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .user()
            .search()
            .limit(0u32)
            .build()?
            .send()
            .await
            .expect_err("expected error");

        if let Error::Api(errors) = res {
            assert_eq!(errors.errors.len(), 1);

            assert_eq!(errors.errors[0].id, error_id);
            assert_eq!(errors.errors[0].status, 400);
            assert_eq!(errors.errors[0].title, Some("Invalid limit".to_string()));
            assert_eq!(
                errors.errors[0].detail,
                Some("Limit must be between 1 and 100".to_string())
            );
        }

        Ok(())
    }
}
