//! Builder for the scanlation group update endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/put-group-id>
//!
//! # Examples
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::{MangaDexClient, Password, Username};
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let _login_res = client
//!     .auth()
//!     .login()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter23")?)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let group_id = Uuid::new_v4();
//! let res = client
//!     .scanlation_group()
//!     .update()
//!     .group_id(&group_id)
//!     .version(2u32)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("update: {:?}", res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::v5::schema::GroupResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned")]
pub struct UpdateGroup<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[serde(skip)]
    pub group_id: &'a Uuid,

    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub name: Option<&'a str>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub leader: Option<&'a Uuid>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub website: Option<&'a str>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub irc_server: Option<&'a str>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub irc_channel: Option<&'a str>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub discord: Option<&'a str>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub contact_email: Option<&'a str>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub description: Option<&'a str>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub locked: Option<bool>,
    pub version: u32,
}

endpoint! {
    PUT ("/group/{}", group_id),
    #[body auth] UpdateGroup<'_>,
    #[flatten_result] GroupResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{body_json, header, method, path_regex};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::v5::AuthTokens;
    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn update_group_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let group_id = Uuid::new_v4();
        let date_time: DateTime<Utc> = DateTime().fake();
        let expected_body = json!({
            "version": 2
        });
        let response_body = json!({
            "result": "ok",
            "data": {
                "id": group_id,
                "type": "scanlation_group",
                "attributes": {
                    "name": "Scanlation Group",
                    "website": null,
                    "ircServer": null,
                    "ircChannel": null,
                    "discord": null,
                    "contactEmail": null,
                    "description": null,
                    "locked": false,
                    "official": false,
                    "verified": false,
                    "version": 2,
                    "createdAt": date_time,
                    "updatedAt": date_time
                }
            },
            "relationships": []
        });

        Mock::given(method("PUT"))
            .and(path_regex(r"/group/[0-9a-fA-F-]+"))
            .and(header("Authorization", "Bearer sessiontoken"))
            .and(header("Content-Type", "application/json"))
            .and(body_json(expected_body))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let _ = mangadex_client
            .scanlation_group()
            .update()
            .group_id(&group_id)
            .version(2u32)
            .build()?
            .send()
            .await?;

        Ok(())
    }
}
