//! Builder for the scanlation group list endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-search-group>
//!
//! # Examples
//!
//! ```rust
//! use mangadex_api::types::MangaStatus;
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let group_res = client
//!     .scanlation_group()
//!     .list()
//!     .name("mangadex")
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("groups: {:?}", group_res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::types::RelationshipType;
use crate::v5::schema::GroupListResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder, Default)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), default, pattern = "owned")]
pub struct ListGroup {
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    pub limit: Option<u32>,
    pub offset: Option<u32>,
    #[builder(setter(each = "add_group_id"))]
    #[serde(rename = "ids")]
    pub group_ids: Vec<Uuid>,
    pub name: Option<String>,
    #[builder(setter(each = "add_relationship"))]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET "/group",
    #[query] ListGroup,
    GroupListResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::{Error, HttpClient, MangaDexClient};

    #[tokio::test]
    async fn list_manga_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let group_id = Uuid::new_v4();
        let date_time: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "results": [
                {
                    "result": "ok",
                    "data": {
                        "id": group_id,
                        "type": "scanlation_group",
                        "attributes": {
                            "name": "Scanlation Group",
                            "website": "https://example.org",
                            "ircServer": null,
                            "ircChannel": null,
                            "discord": null,
                            "contactEmail": null,
                            "description": null,
                            "locked": false,
                            "official": false,
                            "verified": false,
                            "version": 1,
                            "createdAt": date_time,
                            "updatedAt": date_time
                        }
                    },
                    "relationships": []
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        });

        Mock::given(method("GET"))
            .and(path("/group"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .scanlation_group()
            .search()
            .limit(1u32)
            .build()?
            .send()
            .await?;

        let group = res.results[0].as_ref().unwrap();
        assert_eq!(group.data.id, group_id);
        assert_eq!(group.data.attributes.name, "Scanlation Group");
        assert_eq!(
            group.data.attributes.website,
            Some("https://example.org".to_string())
        );
        assert_eq!(group.data.attributes.irc_server, None);
        assert_eq!(group.data.attributes.irc_channel, None);
        assert_eq!(group.data.attributes.discord, None);
        assert_eq!(group.data.attributes.contact_email, None);
        assert_eq!(group.data.attributes.description, None);
        assert_eq!(group.data.attributes.locked, false);
        assert_eq!(group.data.attributes.version, 1);
        assert_eq!(group.data.attributes.created_at, date_time);
        assert_eq!(group.data.attributes.updated_at, date_time);

        Ok(())
    }

    #[tokio::test]
    #[ignore = "Handling list endpoint errors needs work/refactoring"]
    async fn list_manga_handles_400() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let error_id = Uuid::new_v4();

        let response_body = json!({
            "result": "error",
            "errors": [{
                "id": error_id.to_string(),
                "status": 400,
                "title": "Invalid limit",
                "detail": "Limit must be between 1 and 100"
            }]
        });

        Mock::given(method("GET"))
            .and(path("/group"))
            .respond_with(ResponseTemplate::new(400).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .scanlation_group()
            .search()
            .limit(0u32)
            .build()?
            .send()
            .await
            .expect_err("expected error");

        if let Error::Api(errors) = res {
            assert_eq!(errors.errors.len(), 1);

            assert_eq!(errors.errors[0].id, error_id);
            assert_eq!(errors.errors[0].status, 400);
            assert_eq!(errors.errors[0].title, Some("Invalid limit".to_string()));
            assert_eq!(
                errors.errors[0].detail,
                Some("Limit must be between 1 and 100".to_string())
            );
        }

        Ok(())
    }
}
