//! Builder for the scanlation group creation endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/post-group>
//!
//! # Examples
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::{MangaDexClient, Password, Username};
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let _login_res = client
//!     .auth()
//!     .login()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter23")?)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let res = client
//!     .scanlation_group()
//!     .create()
//!     .name("Author Name")
//!     .version(1u32)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("scanlation group create: {:?}", res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::v5::schema::GroupResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned")]
pub struct CreateGroup<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    pub name: &'a str,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub leader: Option<&'a Uuid>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[builder(default)]
    pub members: Option<Vec<&'a Uuid>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub version: Option<u32>,
}

endpoint! {
    POST ("/group"),
    #[body auth] CreateGroup<'_>,
    #[flatten_result] GroupResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::faker::name::en::Name;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{header, method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::v5::AuthTokens;
    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn create_author_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let group_id = Uuid::new_v4();
        let group_name: String = Name().fake();
        let date_time: DateTime<Utc> = DateTime().fake();
        let _expected_body = json!({
            "name": group_name,
            "version": 1
        });
        let response_body = json!({
            "result": "ok",
            "data": {
                "id": group_id,
                "type": "scanlation_group",
                "attributes": {
                    "name": group_name,
                    "website": null,
                    "ircServer": null,
                    "ircChannel": null,
                    "discord": null,
                    "contactEmail": null,
                    "description": null,
                    "locked": false,
                    "official": false,
                    "verified": false,
                    "version": 1,
                    "createdAt": date_time,
                    "updatedAt": date_time
                }
            },
            "relationships": []
        });

        Mock::given(method("POST"))
            .and(path(r"/group"))
            .and(header("Authorization", "Bearer sessiontoken"))
            .and(header("Content-Type", "application/json"))
            // TODO: Make the request body check work.
            // .and(body_json(expected_body))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let _ = mangadex_client
            .scanlation_group()
            .create()
            .name(group_name.as_str())
            .version(1u32)
            .build()?
            .send()
            .await?;

        Ok(())
    }
}
