//! Builder for the create manga endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/post-manga>
//!
//! ```rust
//! use std::collections::HashMap;
//!
//! use mangadex_api::v5::{MangaDexClient, Username, Password};
//! use mangadex_api::types::Language;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let _login_res = client
//!     .auth()
//!     .login()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter23")?)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let manga_res = client
//!     .manga()
//!     .create()
//!     .add_title((Language::English, "My New Manga Title".to_string()))
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("Manga creation: {:?}", manga_res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::{Deserialize, Serialize};
use uuid::Uuid;

use crate::types::{ContentRating, Demographic, Language, MangaLinks, MangaStatus, Tag};
use crate::v5::schema::{LocalizedString, MangaResponse};
use crate::HttpClientRef;

/// Create a new manga.
///
/// This requires authentication.
///
/// Makes a request to `POST /manga`.
#[derive(Debug, Builder, Serialize, Deserialize, Clone)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option))]
pub struct CreateManga {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[builder(setter(each = "add_title"))]
    pub title: LocalizedString,
    #[builder(default)]
    pub alt_titles: Option<Vec<LocalizedString>>,
    #[builder(default)]
    pub description: Option<LocalizedString>,
    #[builder(default)]
    pub authors: Option<Vec<Uuid>>,
    #[builder(default)]
    pub artists: Option<Vec<Uuid>>,
    #[builder(default)]
    pub links: Option<MangaLinks>,
    pub original_language: Language,
    #[builder(default)]
    pub last_volume: Option<String>,
    #[builder(default)]
    pub last_chapter: Option<String>,
    #[builder(default)]
    pub publication_demographic: Option<Demographic>,
    pub status: MangaStatus,
    /// Year the manga was released.
    #[builder(default)]
    pub year: Option<u16>,
    pub content_rating: ContentRating,
    #[builder(default)]
    pub tags: Option<Vec<Tag>>,
    #[builder(default)]
    pub mod_notes: Option<String>,
    pub version: u32,
}

endpoint! {
    POST "/manga",
    #[body auth] CreateManga,
    #[flatten_result] MangaResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{header, method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::types::{ContentRating, Demographic, Language, MangaStatus, Tag};
    use crate::v5::AuthTokens;
    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn create_manga_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let manga_id = Uuid::new_v4();
        let tag_id: Uuid = Tag::Action.into();
        let manga_title = "Test Manga".to_string();
        let created_at: DateTime<Utc> = DateTime().fake();
        let _expected_body = json!({
            "title": {
                "en": manga_title
            },
            "originalLanguage": "ja",
            "publicationDemographic": "shounen",
            "status": "ongoing",
            "contentRating": "safe",
            "tags": [tag_id]

        });
        let response_body = json!({
            "result": "ok",
            "data": {
                "id": manga_id,
                "type": "manga",
                "attributes": {
                    "title": {
                        "en": manga_title
                    },
                    "altTitles": [],
                    "description": {},
                    "isLocked": false,
                    "links": null,
                    "originalLanguage": "ja",
                    "lastVolume": null,
                    "lastChapter": null,
                    "publicationDemographic": "shounen",
                    "status": "ongoing",
                    "year": null,
                    "contentRating": "safe",
                    "tags": [
                        {
                            "id": tag_id,
                            "type": "tag",
                            "attributes": {
                                "name": {
                                    "en": "Action"
                                },
                                "description": [],
                                "group": "genre",
                                "version": 1
                            }
                        }
                    ],
                    "createdAt": created_at,
                    "updatedAt": created_at,
                    "version": 1
                }
            },
            "relationships": []
        });

        Mock::given(method("POST"))
            .and(path("/manga"))
            .and(header("Authorization", "Bearer sessiontoken"))
            .and(header("Content-Type", "application/json"))
            // TODO: Make the request body check work.
            // .and(body_json(expected_body))
            .respond_with(ResponseTemplate::new(201).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .manga()
            .create()
            .add_title((Language::English, manga_title.clone()))
            .original_language(Language::Japanese)
            .publication_demographic(Demographic::Shounen)
            .status(MangaStatus::Ongoing)
            .content_rating(ContentRating::Safe)
            .tags(vec![Tag::Action])
            .version(1u16)
            .build()?
            .send()
            .await?;

        assert_eq!(res.data.id, manga_id);
        assert_eq!(
            res.data.attributes.title.get(&Language::English).unwrap(),
            &manga_title
        );
        assert!(res.data.attributes.alt_titles.is_empty());
        assert!(res.data.attributes.description.is_empty());
        assert!(!res.data.attributes.is_locked);
        assert_eq!(res.data.attributes.links, None);
        assert_eq!(res.data.attributes.original_language, Language::Japanese);
        assert_eq!(res.data.attributes.last_volume, None);
        assert_eq!(res.data.attributes.last_chapter, None);
        assert_eq!(
            res.data.attributes.publication_demographic.unwrap(),
            Demographic::Shounen
        );
        assert_eq!(res.data.attributes.status, MangaStatus::Ongoing);
        assert_eq!(res.data.attributes.year, None);
        assert_eq!(
            res.data.attributes.content_rating.unwrap(),
            ContentRating::Safe
        );
        assert_eq!(
            res.data.attributes.tags[0]
                .attributes
                .name
                .get(&Language::English),
            Some(&"Action".to_string())
        );
        assert_eq!(res.data.attributes.created_at, created_at);
        assert_eq!(res.data.attributes.updated_at.unwrap(), created_at);
        assert_eq!(res.data.attributes.version, 1);

        Ok(())
    }
}
