//! Builder for the chapter list endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-chapter>
//!
//! # Examples
//!
//! ```rust
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let chapter_res = client
//!     .chapter()
//!     .list()
//!     .title("summoning")
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("chapters: {:?}", chapter_res);
//! # Ok(())
//! # }
//! ```

use chrono::{DateTime, Utc};
use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::types::{ChapterSortOrder, ContentRating, Language, RelationshipType};
use crate::v5::schema::ChapterListResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder, Default)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), default, pattern = "owned")]
pub struct ListChapter<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    pub limit: Option<u32>,
    pub offset: Option<u32>,
    #[serde(rename = "ids")]
    #[builder(setter(each = "add_chapter_id"))]
    pub chapter_ids: Vec<&'a Uuid>,
    pub title: Option<String>,
    #[builder(setter(each = "add_group"))]
    pub groups: Vec<&'a Uuid>,
    pub uploader: Option<&'a Uuid>,
    #[serde(rename = "manga")]
    pub manga_id: Option<&'a Uuid>,
    #[serde(rename = "volume")]
    #[builder(setter(each = "add_volume"))]
    pub volumes: Vec<String>,
    /// Chapter number in the series or volume.
    #[builder(setter(each = "add_chapter"))]
    pub chapters: Vec<String>,
    #[serde(rename = "translatedLanguage")]
    #[builder(setter(each = "add_translated_language"))]
    pub translated_languages: Vec<Language>,
    #[serde(rename = "originalLanguage")]
    #[builder(setter(each = "add_original_language"))]
    pub original_languages: Vec<Language>,
    #[serde(rename = "excludedOriginalLanguage")]
    #[builder(setter(each = "exclude_original_language"))]
    pub excluded_original_languages: Vec<Language>,
    #[builder(setter(each = "add_content_rating"))]
    pub content_rating: Vec<ContentRating>,
    pub created_at_since: Option<DateTime<Utc>>,
    pub updated_at_since: Option<DateTime<Utc>>,
    pub publish_at_since: Option<DateTime<Utc>>,
    pub order: Option<ChapterSortOrder>,
    #[builder(setter(each = "add_relationship"))]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET "/chapter",
    #[query] ListChapter<'_>,
    ChapterListResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::faker::internet::en::Password;
    use fake::faker::name::en::Name;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::types::Language;
    use crate::{Error, HttpClient, MangaDexClient};

    #[tokio::test]
    async fn list_chapter_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let chapter_id = Uuid::new_v4();
        let uploader_id = Uuid::new_v4();
        let chapter_title: String = Name().fake();
        let hash: String = Password(16..24).fake();
        let created_at: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "results": [
                {
                    "result": "ok",
                    "data": {
                        "id": chapter_id,
                        "type": "chapter",
                        "attributes": {
                            "title": chapter_title,
                            "volume": "1",
                            "chapter": "1.5",
                            "translatedLanguage": "en",
                            "hash": hash,
                            "data": [
                                "1.jpg"
                            ],
                            "dataSaver": [
                                "1.jpg"
                            ],
                            "uploader": uploader_id,
                            "version": 1,
                            "createdAt": created_at,
                            "updatedAt": created_at,
                            "publishAt": created_at
                        }
                    },
                    "relationships": []
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        });

        Mock::given(method("GET"))
            .and(path("/chapter"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .chapter()
            .search()
            .limit(1u32)
            .build()?
            .send()
            .await?;

        if let Ok(chapter) = &res.results[0] {
            assert_eq!(chapter.data.id, chapter_id);
            assert_eq!(chapter.data.attributes.title, chapter_title);
            assert_eq!(chapter.data.attributes.volume, Some("1".to_string()));
            assert_eq!(chapter.data.attributes.chapter, Some("1.5".to_string()));
            assert_eq!(
                chapter.data.attributes.translated_language,
                Language::English
            );
            assert_eq!(chapter.data.attributes.hash, hash);
            assert_eq!(chapter.data.attributes.data, vec!["1.jpg"]);
            assert_eq!(chapter.data.attributes.data_saver, vec!["1.jpg"]);
            assert_eq!(chapter.data.attributes.uploader, Some(uploader_id));
            assert_eq!(chapter.data.attributes.version, 1);
            assert_eq!(chapter.data.attributes.created_at, created_at);
            assert_eq!(chapter.data.attributes.updated_at, Some(created_at));
            assert_eq!(chapter.data.attributes.publish_at, created_at);
        }

        Ok(())
    }

    #[tokio::test]
    #[ignore = "Handling list endpoint errors needs work/refactoring"]
    async fn list_chapter_handles_400() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let error_id = Uuid::new_v4();

        let response_body = json!({
            "result": "error",
            "errors": [{
                "id": error_id.to_string(),
                "status": 400,
                "title": "Invalid limit",
                "detail": "Limit must be between 1 and 100"
            }]
        });

        Mock::given(method("GET"))
            .and(path("/chapter"))
            .respond_with(ResponseTemplate::new(400).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .chapter()
            .search()
            .limit(0u32)
            .build()?
            .send()
            .await
            .expect_err("expected error");

        if let Error::Api(errors) = res {
            assert_eq!(errors.errors.len(), 1);

            assert_eq!(errors.errors[0].id, error_id);
            assert_eq!(errors.errors[0].status, 400);
            assert_eq!(errors.errors[0].title, Some("Invalid limit".to_string()));
            assert_eq!(
                errors.errors[0].detail,
                Some("Limit must be between 1 and 100".to_string())
            );
        }

        Ok(())
    }
}
