use std::str::FromStr;
use std::string::ParseError;

use serde::{Deserialize, Serialize};

/// Languages supported by MangaDex.
#[derive(Clone, Copy, Debug, Deserialize, Eq, Hash, PartialEq, PartialOrd, Serialize)]
pub enum Language {
    #[serde(rename = "ar")]
    Arabic,
    #[serde(rename = "bn")]
    Bengali,
    #[serde(rename = "bg")]
    Bulgarian,
    #[serde(rename = "my")]
    Burmese,
    #[serde(rename = "ca")]
    Catalan,
    #[serde(rename = "zh")]
    ChineseSimplified,
    #[serde(rename = "zh-hk")]
    ChineseTraditional,
    #[serde(rename = "cs")]
    Czech,
    #[serde(rename = "da")]
    Danish,
    #[serde(rename = "nl")]
    Dutch,
    #[serde(rename = "en")]
    English,
    #[serde(rename = "tl")]
    Filipino,
    #[serde(rename = "fi")]
    Finnish,
    #[serde(rename = "fr")]
    French,
    #[serde(rename = "de")]
    German,
    #[serde(rename = "el")]
    Greek,
    #[serde(rename = "he")]
    Hebrew,
    #[serde(rename = "hi")]
    Hindi,
    #[serde(rename = "hu")]
    Hungarian,
    #[serde(rename = "id")]
    Indonesian,
    #[serde(rename = "it")]
    Italian,
    #[serde(rename = "ja")]
    Japanese,
    #[serde(rename = "ko")]
    Korean,
    #[serde(rename = "lt")]
    Lithuanian,
    #[serde(rename = "ms")]
    Malay,
    #[serde(rename = "mn")]
    Mongolian,
    #[serde(rename = "kr")]
    NiloSaharan,
    #[serde(rename = "no")]
    Norwegian,
    #[serde(rename = "fa")]
    Persian,
    #[serde(rename = "pl")]
    Polish,
    #[serde(rename = "pt-br")]
    PortugueseBrazil,
    #[serde(rename = "pt")]
    PortuguesePortugal,
    #[serde(rename = "ro")]
    Romanian,
    #[serde(rename = "ru")]
    Russian,
    #[serde(rename = "sr")]
    SerboCroatian,
    #[serde(rename = "es")]
    SpanishEs,
    #[serde(rename = "es-la")]
    SpanishLATAM,
    #[serde(rename = "sv")]
    Swedish,
    #[serde(rename = "th")]
    Thai,
    #[serde(rename = "tr")]
    Turkish,
    #[serde(rename = "uk")]
    Ukrainian,
    #[serde(rename = "vi")]
    Vietnamese,
    #[serde(rename = "NULL")]
    Null,
}

impl Language {
    /// Get the ISO 639-1 2-letter code representation.
    pub fn code2(&self) -> &str {
        match self {
            Self::Arabic => "ar",
            Self::Bengali => "bn",
            Self::Bulgarian => "bg",
            Self::Burmese => "my",
            Self::Catalan => "ca",
            Self::ChineseSimplified => "zh",
            Self::ChineseTraditional => "zh-hk",
            Self::Czech => "cs",
            Self::Danish => "da",
            Self::Dutch => "nl",
            Self::English => "en",
            Self::Filipino => "tl",
            Self::Finnish => "fi",
            Self::French => "fr",
            Self::German => "de",
            Self::Greek => "el",
            Self::Hebrew => "he",
            Self::Hindi => "hi",
            Self::Hungarian => "hu",
            Self::Indonesian => "id",
            Self::Italian => "it",
            Self::Japanese => "ja",
            Self::Korean => "ko",
            Self::Lithuanian => "lt",
            Self::Malay => "ms",
            Self::Mongolian => "mn",
            Self::NiloSaharan => "kr",
            Self::Norwegian => "no",
            Self::Persian => "fa",
            Self::Polish => "pl",
            Self::PortugueseBrazil => "pt-br",
            Self::PortuguesePortugal => "pt",
            Self::Romanian => "rm",
            Self::Russian => "ru",
            Self::SerboCroatian => "sr",
            Self::SpanishEs => "es",
            Self::SpanishLATAM => "es-la",
            Self::Swedish => "sv",
            Self::Thai => "th",
            Self::Turkish => "tr",
            Self::Ukrainian => "uk",
            Self::Vietnamese => "vi",
            Self::Null => "NULL",
        }
    }
}

impl From<&str> for Language {
    /// Parse a `Language` type from a string.
    ///
    /// This function's value parameter is case-insensitive.
    fn from(value: &str) -> Self {
        match value.to_lowercase().as_str() {
            "ar" => Self::Arabic,
            "bn" => Self::Bengali,
            "bg" => Self::Bulgarian,
            "my" => Self::Burmese,
            "ca" => Self::Catalan,
            "zh" => Self::ChineseSimplified,
            "zh-hk" => Self::ChineseTraditional,
            "cs" => Self::Czech,
            "da" => Self::Danish,
            "nl" => Self::Dutch,
            "en" => Self::English,
            "tl" => Self::Filipino,
            "fi" => Self::Finnish,
            "fr" => Self::French,
            "de" => Self::German,
            "el" => Self::Greek,
            "he" => Self::Hebrew,
            "hi" => Self::Hindi,
            "hu" => Self::Hungarian,
            "id" => Self::Indonesian,
            "it" => Self::Italian,
            "ja" => Self::Japanese,
            "ko" => Self::Korean,
            "lt" => Self::Lithuanian,
            "ms" => Self::Malay,
            "mn" => Self::Mongolian,
            "kr" => Self::NiloSaharan,
            "no" => Self::Norwegian,
            "fa" => Self::Persian,
            "pl" => Self::Polish,
            "pt-br" => Self::PortugueseBrazil,
            "pt" => Self::PortuguesePortugal,
            "rm" => Self::Romanian,
            "ru" => Self::Russian,
            "sr" => Self::SerboCroatian,
            "es" => Self::SpanishEs,
            "es-la" => Self::SpanishLATAM,
            "sv" => Self::Swedish,
            "th" => Self::Thai,
            "tr" => Self::Turkish,
            "uk" => Self::Ukrainian,
            "vi" => Self::Vietnamese,
            _ => Self::Null,
        }
    }
}

impl FromStr for Language {
    type Err = ParseError;

    /// Parse a `Language` type from a string.
    ///
    /// This function's value parameter is case-insensitive.
    fn from_str(value: &str) -> Result<Self, ParseError> {
        Ok(match value.to_lowercase().as_str() {
            "ar" => Self::Arabic,
            "bn" => Self::Bengali,
            "bg" => Self::Bulgarian,
            "my" => Self::Burmese,
            "ca" => Self::Catalan,
            "zh" => Self::ChineseSimplified,
            "zh-hk" => Self::ChineseTraditional,
            "cs" => Self::Czech,
            "da" => Self::Danish,
            "nl" => Self::Dutch,
            "en" => Self::English,
            "tl" => Self::Filipino,
            "fi" => Self::Finnish,
            "fr" => Self::French,
            "de" => Self::German,
            "el" => Self::Greek,
            "he" => Self::Hebrew,
            "hi" => Self::Hindi,
            "hu" => Self::Hungarian,
            "id" => Self::Indonesian,
            "it" => Self::Italian,
            "ja" => Self::Japanese,
            "ko" => Self::Korean,
            "lt" => Self::Lithuanian,
            "ms" => Self::Malay,
            "mn" => Self::Mongolian,
            "kr" => Self::NiloSaharan,
            "no" => Self::Norwegian,
            "fa" => Self::Persian,
            "pl" => Self::Polish,
            "pt-br" => Self::PortugueseBrazil,
            "pt" => Self::PortuguesePortugal,
            "rm" => Self::Romanian,
            "ru" => Self::Russian,
            "sr" => Self::SerboCroatian,
            "es" => Self::SpanishEs,
            "es-la" => Self::SpanishLATAM,
            "sv" => Self::Swedish,
            "th" => Self::Thai,
            "tr" => Self::Turkish,
            "uk" => Self::Ukrainian,
            "vi" => Self::Vietnamese,
            _ => Self::Null,
        })
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn string_produces_english_from_en() {
        let lang = Language::from("en");
        assert_eq!(lang, Language::English);
    }

    #[test]
    fn string_produces_japanese_from_capitalized_ja() {
        let lang = Language::from("JA");
        assert_eq!(lang, Language::Japanese);
    }

    #[test]
    fn string_produces_null_from_unknown() {
        let test_cases = ["foo", "bar", "baz"];
        for test in test_cases {
            let lang = Language::from(test);
            assert_eq!(lang, Language::Null);
        }
    }
}
