//! Report endpoint handler.
//!
//! <https://api.mangadex.org/docs.html#tag/Report>

mod create;
mod list;

use std::cell::RefCell;
use std::rc::Rc;

use crate::v5::report::create::CreateReportBuilder;
use crate::v5::report::list::ListReasonsBuilder;
use crate::v5::HttpClient;

/// Report endpoint handler builder.
#[derive(Clone, Debug)]
pub struct ReportBuilder {
    http_client: Rc<RefCell<HttpClient>>,
}

impl ReportBuilder {
    pub fn new(http_client: Rc<RefCell<HttpClient>>) -> Self {
        Self { http_client }
    }

    /// Get a list of report reasons.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-report-reasons-by-category>
    pub fn list(&self) -> ListReasonsBuilder {
        ListReasonsBuilder::default().http_client(self.http_client.clone())
    }

    /// Create a new report.
    ///
    /// <https://api.mangadex.org/docs.html#operation/post-report>
    pub fn create(&self) -> CreateReportBuilder {
        CreateReportBuilder::default().http_client(self.http_client.clone())
    }
}
