//! Builder for the manga view endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-manga-id>
//!
//! # Examples
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let res = client
//!     .manga()
//!     .list_tags()
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("tags: {:?}", res);
//! # Ok(())
//! # }
//! ```

use std::cell::RefCell;
use std::rc::Rc;

use derive_builder::Builder;
use serde::Serialize;

use crate::v5::schema::TagListResponse;
use crate::v5::HttpClient;

#[derive(Debug, Serialize, Clone, Builder)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned")]
pub struct ListTags {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: Rc<RefCell<HttpClient>>,
}

endpoint! {
    GET "/manga/tag",
    #[no_data] ListTags,
    TagListResponse
}

#[cfg(test)]
mod tests {
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::types::{Language, TagGroup};
    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn get_manga_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let response_body = json!([
            {
                "result": "ok",
                "data": {
                    "id": "0234a31e-a729-4e28-9d6a-3f87c4966b9e",
                    "type": "tag",
                    "attributes": {
                        "name": {
                            "en": "Oneshot"
                        },
                        "description": [],
                        "group": "format",
                        "version": 1
                    }
                },
                "relationships": []
            },
            {
                "result": "ok",
                "data": {
                    "id": "07251805-a27e-4d59-b488-f0bfbec15168",
                    "type": "tag",
                    "attributes": {
                        "name": {
                            "en": "Thriller"
                        },
                        "description": [],
                        "group": "genre",
                        "version": 1
                    }
                },
                "relationships": []
            }
        ]);

        Mock::given(method("GET"))
            .and(path(r"/manga/tag"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client.manga().list_tags().build()?.send().await?;

        let oneshot = res[0].as_ref().unwrap();
        assert_eq!(
            oneshot.data.id,
            Uuid::parse_str("0234a31e-a729-4e28-9d6a-3f87c4966b9e")?
        );
        assert_eq!(
            oneshot.data.attributes.name.get(&Language::English),
            Some(&"Oneshot".to_string())
        );
        assert_eq!(oneshot.data.attributes.group, TagGroup::Format);

        // Skip the second tag checks for now.
        // If the first result passes, the second probably will as well.

        Ok(())
    }
}
