//! Builder for the manga list endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-search-manga>
//!
//! # Examples
//!
//! ```rust
//! use mangadex_api::types::MangaStatus;
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let manga_res = client
//!     .manga()
//!     .list()
//!     .title("full metal")
//!     .add_status(MangaStatus::Completed)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("manga: {:?}", manga_res);
//! # Ok(())
//! # }
//! ```

use std::cell::RefCell;
use std::rc::Rc;

use chrono::{DateTime, Utc};
use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::types::{
    ContentRating, Demographic, Language, MangaSortOrder, MangaStatus, RelationshipType, Tag,
    TagSearchMode,
};
use crate::v5::schema::MangaListResponse;
use crate::v5::HttpClient;

#[derive(Debug, Serialize, Clone, Builder, Default)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), default, pattern = "owned")]
pub struct ListManga {
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: Rc<RefCell<HttpClient>>,

    pub limit: Option<u32>,
    pub offset: Option<u32>,
    pub title: Option<String>,
    #[builder(setter(each = "add_author"))]
    pub authors: Vec<Uuid>,
    #[builder(setter(each = "add_artist"))]
    pub artists: Vec<Uuid>,
    pub year: Option<u16>,
    #[builder(setter(each = "include_tag"))]
    pub included_tags: Vec<Tag>,
    pub included_tags_mode: Option<TagSearchMode>,
    #[builder(setter(each = "exclude_tag"))]
    pub excluded_tags: Vec<Tag>,
    pub excluded_tags_mode: Option<TagSearchMode>,
    #[builder(setter(each = "add_status"))]
    pub status: Vec<MangaStatus>,
    #[builder(setter(each = "add_original_language"))]
    pub original_language: Vec<Language>,
    #[builder(setter(each = "add_publication_demographic"))]
    pub publication_demographic: Vec<Demographic>,
    #[builder(setter(each = "add_manga_id"))]
    #[serde(rename = "ids")]
    pub manga_ids: Vec<Uuid>,
    #[builder(setter(each = "add_content_rating"))]
    pub content_rating: Vec<ContentRating>,
    pub created_at_since: Option<DateTime<Utc>>,
    pub updated_at_since: Option<DateTime<Utc>>,
    pub order: Option<MangaSortOrder>,
    #[builder(setter(each = "add_relationship"))]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET "/manga",
    #[query] ListManga,
    MangaListResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::types::{ContentRating, Demographic, Language, MangaStatus};
    use crate::{Error, HttpClient, MangaDexClient};

    #[tokio::test]
    async fn list_manga_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let manga_id = Uuid::new_v4();
        let manga_title = "Test Manga".to_string();
        let created_at: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "results": [
                {
                    "result": "ok",
                    "data": {
                        "id": manga_id,
                        "type": "manga",
                        "attributes": {
                            "title": {
                                "en": manga_title
                            },
                            "altTitles": [],
                            "description": {},
                            "isLocked": false,
                            "links": null,
                            "originalLanguage": "ja",
                            "lastVolume": null,
                            "lastChapter": null,
                            "publicationDemographic": "shoujo",
                            "status": "ongoing",
                            "year": null,
                            "contentRating": "safe",
                            "tags": [],
                            "createdAt": created_at,
                            "updatedAt": created_at,
                            "version": 1
                        }
                    },
                    "relationships": []
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        });

        Mock::given(method("GET"))
            .and(path("/manga"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .manga()
            .search()
            .limit(1u32)
            .build()?
            .send()
            .await?;

        let manga = res.results[0].as_ref().unwrap();
        assert_eq!(manga.data.id, manga_id);
        assert_eq!(
            manga.data.attributes.title.get(&Language::English).unwrap(),
            &manga_title
        );
        assert!(manga.data.attributes.alt_titles.is_empty());
        assert!(manga.data.attributes.description.is_empty());
        assert!(!manga.data.attributes.is_locked);
        assert_eq!(manga.data.attributes.links, None);
        assert_eq!(manga.data.attributes.original_language, Language::Japanese);
        assert_eq!(manga.data.attributes.last_volume, None);
        assert_eq!(manga.data.attributes.last_chapter, None);
        assert_eq!(
            manga.data.attributes.publication_demographic.unwrap(),
            Demographic::Shoujo
        );
        assert_eq!(manga.data.attributes.status.unwrap(), MangaStatus::Ongoing);
        assert_eq!(manga.data.attributes.year, None);
        assert_eq!(
            manga.data.attributes.content_rating.unwrap(),
            ContentRating::Safe
        );
        assert!(manga.data.attributes.tags.is_empty());
        assert_eq!(manga.data.attributes.created_at, created_at);
        assert_eq!(manga.data.attributes.updated_at.unwrap(), created_at);
        assert_eq!(manga.data.attributes.version, 1);

        Ok(())
    }

    #[tokio::test]
    #[ignore = "Handling list endpoint errors needs work/refactoring"]
    async fn list_manga_handles_400() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let error_id = Uuid::new_v4();

        let response_body = json!({
            "result": "error",
            "errors": [{
                "id": error_id.to_string(),
                "status": 400,
                "title": "Invalid limit",
                "detail": "Limit must be between 1 and 100"
            }]
        });

        Mock::given(method("GET"))
            .and(path("/manga"))
            .respond_with(ResponseTemplate::new(400).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .manga()
            .search()
            .limit(0u32)
            .build()?
            .send()
            .await
            .expect_err("expected error");

        if let Error::Api(errors) = res {
            assert_eq!(errors.errors.len(), 1);

            assert_eq!(errors.errors[0].id, error_id);
            assert_eq!(errors.errors[0].status, 400);
            assert_eq!(errors.errors[0].title, Some("Invalid limit".to_string()));
            assert_eq!(
                errors.errors[0].detail,
                Some("Limit must be between 1 and 100".to_string())
            );
        }

        Ok(())
    }
}
