use crate::v5::schema::MangaDexErrorResponse;

pub type Result<T, E = Error> = std::result::Result<T, E>;

#[derive(Debug, thiserror::Error)]
pub enum Error {
    /// Error when parsing a URL.
    ///
    /// This should not happen.
    #[error("Error parsing the URL.")]
    ParseUrlError(#[from] url::ParseError),

    #[error("Failed to send a request to MangaDex.")]
    RequestError(#[from] reqwest::Error),

    #[error("Missing auth tokens. Please log in to MangaDex.")]
    MissingTokens,

    #[error("Not a valid username.")]
    UsernameError(String),

    #[error("Not a valid password.")]
    PasswordError(String),

    #[error("An error occurred while pinging the MangaDex server.")]
    PingError,

    /// Errors returned from the MangaDex API request.
    #[error("An error occurred with the MangaDex API request.")]
    Api(#[from] MangaDexErrorResponse),

    /// Error while building the request struct.
    #[error("An error occurred while building the request.")]
    RequestBuilderError(String),

    /// Error while parsing the type.
    #[error("An error occurred while parsing the type.")]
    ParseError(String),

    #[error(transparent)]
    UnexpectedError(#[from] anyhow::Error),
}
