//! Builder for uploading a new manga cover.
//!
//! <https://api.mangadex.org/docs.html#operation/upload-cover>
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::{MangaDexClient, Username, Password};
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let _login_res = client
//!     .auth()
//!     .login()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter23")?)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let manga_id = Uuid::new_v4();
//! let file_bytes = vec![0];
//! let res = client
//!     .cover()
//!     .upload()
//!     .manga_id(&manga_id)
//!     .file(file_bytes)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("upload cover: {:?}", res);
//! # Ok(())
//! # }
//! ```

use std::borrow::Cow;
use std::cell::RefCell;
use std::rc::Rc;

use derive_builder::Builder;
use reqwest::multipart::{Form, Part};
use serde::Serialize;
use uuid::Uuid;

use crate::common::Endpoint;
use crate::v5::schema::CoverResponse;
use crate::v5::HttpClient;

/// Upload a new cover for a manga.
///
/// This requires authentication.
///
/// Makes a request to `POST /cover/{id}`.
#[derive(Debug, Builder, Serialize, Clone)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned")]
pub struct UploadCover<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: Rc<RefCell<HttpClient>>,

    #[serde(skip)]
    pub manga_id: &'a Uuid,

    /// Image bytes.
    pub file: Cow<'static, [u8]>,
}

// TODO: Come up with a way to generalize multipart form data for the `Endpoint` trait.
impl Endpoint for UploadCover<'_> {
    type Query = ();
    type Body = ();
    type Response = CoverResponse;

    fn path(&self) -> Cow<str> {
        Cow::Owned(format!("/cover/{}", self.manga_id))
    }

    fn method(&self) -> reqwest::Method {
        reqwest::Method::POST
    }

    fn require_auth(&self) -> bool {
        true
    }

    fn multipart(&self) -> Option<Form> {
        let part = Part::bytes(self.file.clone());
        let form = Form::new().part("file", part);

        Some(form)
    }
}

impl UploadCover<'_> {
    pub async fn send(&self) -> CoverResponse {
        self.http_client.borrow().send_request(self).await?
    }
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::faker::lorem::en::Sentence;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{header, header_exists, method, path_regex};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::v5::AuthTokens;
    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn upload_cover_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let manga_id = Uuid::new_v4();
        let file_bytes = vec![0_u8];
        let cover_id = Uuid::new_v4();
        let description: String = Sentence(1..3).fake();
        let created_at: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "result": "ok",
            "data": {
                "id": cover_id,
                "type": "cover_art",
                "attributes": {
                    "volume": "1",
                    "fileName": "1.jpg",
                    "description": description,
                    "version": 1,
                    "createdAt": created_at,
                    "updatedAt": created_at,
                }
            },
            "relationships": []
        });

        Mock::given(method("POST"))
            .and(path_regex("/cover/[0-9a-fA-F-]+"))
            .and(header("Authorization", "Bearer sessiontoken"))
            // The "multipart/form-data; boundary=[boundary]" Content-Type value is dynamic and can't easily be validated.
            .and(header_exists("Content-Type"))
            .respond_with(ResponseTemplate::new(201).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let _ = mangadex_client
            .upload()
            .cover()
            .manga_id(&manga_id)
            .file(file_bytes)
            .build()?
            .send()
            .await?;

        Ok(())
    }
}
