//! Authentication endpoint handler.
//!
//! <https://api.mangadex.org/docs.html#tag/Auth>

mod check_token;
mod login;
mod logout;
mod refresh_token;

use std::cell::RefCell;
use std::rc::Rc;

use crate::v5::auth::check_token::CheckTokenBuilder;
use crate::v5::auth::login::LoginBuilder;
use crate::v5::auth::logout::LogoutBuilder;
use crate::v5::auth::refresh_token::RefreshTokenBuilder;
use crate::v5::HttpClient;

/// Authentication endpoint handler builder.
#[derive(Debug)]
pub struct AuthBuilder {
    http_client: Rc<RefCell<HttpClient>>,
}

impl AuthBuilder {
    pub fn new(http_client: Rc<RefCell<HttpClient>>) -> Self {
        Self { http_client }
    }

    /// Log into an account.
    ///
    /// <https://api.mangadex.org/docs.html#operation/post-auth-login>
    pub fn login(&self) -> LoginBuilder {
        LoginBuilder::default().http_client(self.http_client.clone())
    }

    /// Log out of an account.
    ///
    /// <https://api.mangadex.org/docs.html#operation/post-auth-logout>
    pub fn logout(&self) -> LogoutBuilder {
        LogoutBuilder::default().http_client(self.http_client.clone())
    }

    /// Get a new session token from the refresh token.
    ///
    /// <https://api.mangadex.org/docs.html#operation/post-auth-refresh>
    pub fn refresh_token(&self) -> RefreshTokenBuilder {
        RefreshTokenBuilder::default().http_client(self.http_client.clone())
    }

    /// Check the current session token and get basic info about the authenticated user.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-auth-check>
    pub fn check_token(&self) -> CheckTokenBuilder {
        CheckTokenBuilder::default().http_client(self.http_client.clone())
    }
}
