//! # mangadex_api
//!
//! The `mangadex_api` crate provides a convenient, high-level wrapper
//! [client] for the [MangaDex API][mangadex-api-url].
//!
//! It aims to cover all public endpoints covered by [their documentation][mangadex-api-url].
//!
//! ## How to install
//!
//! Add `mangadex-api` to your dependencies:
//!
//! ```toml
//! [dependencies]
//! # ...
//! mangadex-api = "1.0.0-alpha.0"
//! ```
//!
//! If you are using [`cargo-edit`](https://github.com/killercup/cargo-edit), run
//!
//! ```bash
//! cargo add mangadex-api
//! ```
//!
//! # Features
//!
//! All features are not included by default. To enable them, add any of the following to your project's `Cargo.toml` file.
//!
//! * `upload` - Enable endpoints that upload files to MangaDex.
//!
//! For example, to enable the `upload` feature, add the following to your `Cargo.toml` file:
//!
//! ```toml
//! mangadex-api = { version = "1.0.0-alpha.0", features = ["upload"] }
//! ```
//!
//! # API Sections
//!
//! The endpoints are grouped into sections, defined by MangaDex, and are reflected in the
//! [`MangaDexClient`][client] methods.
//! The endpoint builders are sub-modules in these categories.
//!
//! - Account
//! - AtHome
//! - Authentication
//! - Author
//! - Captcha
//! - Chapter
//! - Cover
//! - CustomList
//! - Feed
//! - Infrastructure
//! - Legacy
//! - Manga
//! - Report
//! - ScanlationGroup
//! - Search
//! - User
//!
//! # HTTP Client
//!
//! The [`mangadex_api::MangaDexClient`][client] is asynchronous, using
//! [`reqwest`][reqwest] as the HTTP client.
//!
//! ## Using a custom reqwest Client
//!
//! By default, [`mangadex_api::MangaDexClient`][client] will use the default
//! [`reqwest::Client`][reqwest-client] settings.
//!
//! You may provide your own [`reqwest::Client`][reqwest-client] to customize options such as the
//! request timeout.
//!
//! ```rust
//! use reqwest::Client;
//!
//! use mangadex_api::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let reqwest_client = Client::builder()
//!     .timeout(std::time::Duration::from_secs(10))
//!     .build()?;
//!
//! let client = MangaDexClient::new(reqwest_client);
//! # Ok(())
//! # }
//! ```
//!
//! ## Searching manga by title
//!
//! ```rust
//! use mangadex_api::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let manga_results = client
//!     .manga()
//!     .search()
//!     .title("full metal")
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("manga results = {:?}", manga_results);
//! # Ok(())
//! # }
//! ```
//!
//! ## Downloading chapter pages
//!
//! ```rust
//! // Imports used for downloading the pages to a file.
//! // They are not used because we're just printing the raw bytes.
//! // use std::fs::File;
//! // use std::io::Write;
//!
//! use reqwest::Url;
//! use uuid::Uuid;
//!
//! use mangadex_api::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let chapter_id = Uuid::new_v4();
//! let chapter = client
//!     .chapter()
//!     .get()
//!     .chapter_id(&chapter_id)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let at_home_url = client
//!     .at_home()
//!     .server()
//!     .chapter_id(&chapter_id)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let http_client = reqwest::Client::new();
//!
//! // Original quality. Use `.data.attributes.data_saver` for smaller, compressed images.
//! let page_filenames = chapter.data.attributes.data;
//! for filename in page_filenames {
//!     let page_url = at_home_url
//!         .join(&format!(
//!             "/data/{}/{}",
//!             chapter.data.attributes.hash, filename
//!         ))
//!         .unwrap();
//!
//!     let res = http_client.get(page_url).send().await?;
//!     // The data should be streamed rather than downloading the data all at once.
//!     let bytes = res.bytes().await?;
//!
//!     // This is where you would download the file but for this example,
//!     // we're just printing the raw data.
//!     // let mut file = File::create(&filename)?;
//!     // let _ = file.write_all(&bytes);
//!     println!("Chunk: {:?}", bytes);
//! }
//!
//! # Ok(())
//! # }
//! ```
//!
//! ## Downloading a manga's main cover image
//!
//! While this example could directly get the cover information by passing in the cover ID,
//! it is not often that one would have the ID off-hand, so the most common method would be from a
//! manga result.
//!
//! If you want to get all of a manga's cover images, you will need to use the [cover list endpoint](https://api.mangadex.org/docs.html#operation/get-cover)
//! and use the `manga[]` query parameter.
//!
//! ```rust
//! // Imports used for downloading the cover to a file.
//! // They are not used because we're just printing the raw bytes.
//! // use std::fs::File;
//! // use std::io::Write;
//!
//! use reqwest::Url;
//! use uuid::Uuid;
//!
//! use mangadex_api::types::RelationshipType;
//! use mangadex_api::MangaDexClient;
//! use mangadex_api::CDN_URL;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let manga_id = Uuid::new_v4();
//! let manga = client
//!     .manga()
//!     .get()
//!     .manga_id(&manga_id)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let cover_id = manga
//!     .relationships
//!     .iter()
//!     .find(|related| related.type_ == RelationshipType::CoverArt)
//!     .expect("no cover art found for manga")
//!     .id;
//! let cover = client
//!     .cover()
//!     .get()
//!     .cover_id(&cover_id)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! // This uses the best quality image.
//! // To use smaller, thumbnail-sized images, append any of the following:
//! //
//! // - .512.jpg
//! // - .256.jpg
//! //
//! // For example, "https://uploads.mangadex.org/covers/8f3e1818-a015-491d-bd81-3addc4d7d56a/4113e972-d228-4172-a885-cb30baffff97.jpg.512.jpg"
//! let cover_url = Url::parse(&format!(
//!         "{}/covers/{}/{}",
//!         CDN_URL, manga_id, cover.data.attributes.file_name
//!     ))
//!     .unwrap();
//!
//! let http_client = reqwest::Client::new();
//!
//! let res = http_client.get(cover_url).send().await?;
//! // The data should be streamed rather than downloading the data all at once.
//! let bytes = res.bytes().await?;
//!
//! // This is where you would download the file but for this example, we're just printing the raw data.
//! // let mut file = File::create(&filename)?;
//! // let _ = file.write_all(&bytes);
//! println!("Chunk: {:?}", bytes);
//! # Ok(())
//! # }
//! ```
//!
//! [client]: ./v5/struct.MangaDexClient.html
//! [mangadex-api-url]: https://api.mangadex.org
//! [md-account]: https://api.mangadex.org/docs.html#tag/Account
//! [md-auth]: https://api.mangadex.org/docs.html#tag/Auth
//! [md-author]: https://api.mangadex.org/docs.html#tag/Author
//! [md-captcha]: https://api.mangadex.org/docs.html#tag/Captcha
//! [md-chapter]: https://api.mangadex.org/docs.html#tag/Chapter
//! [md-cover]: https://api.mangadex.org/docs.html#tag/Cover
//! [md-custom-list]: https://api.mangadex.org/docs.html#tag/CustomList
//! [md-feed]: https://api.mangadex.org/docs.html#tag/Feed
//! [md-infrastructure]: https://api.mangadex.org/docs.html#tag/Infrastructure
//! [md-legacy]: https://api.mangadex.org/docs.html#tag/Legacy
//! [md-manga]: https://api.mangadex.org/docs.html#tag/Manga
//! [md-md-at-home]: https://api.mangadex.org/docs.html#tag/AtHome
//! [md-report]: https://api.mangadex.org/docs.html#tag/Report
//! [md-scanlation-group]: https://api.mangadex.org/docs.html#tag/ScanlationGroup
//! [md-search]: https://api.mangadex.org/docs.html#tag/Search
//! [md-user]: https://api.mangadex.org/docs.html#tag/User
//! [reqwest]: https://docs.rs/reqwest
//! [reqwest-client]: https://docs.rs/reqwest/latest/reqwest/struct.Client.html
//! [v5-account]: ./v5/account/index.html
//! [v5-auth]: ./v5/auth/index.html
//! [v5-author]: ./v5/author/index.html
//! [v5-captcha]: ./v5/captcha/index.html
//! [v5-chapter]: ./v5/chapter/index.html
//! [v5-cover]: ./v5/cover/index.html
//! [v5-custom-list]: ./v5/custom_list/index.html
//! [v5-feed]: ./v5/feed/index.html
//! [v5-infrastructure]: ./v5/infrastructure/index.html
//! [v5-legacy]: ./v5/legacy/index.html
//! [v5-manga]: ./v5/manga/index.html
//! [v5-md-at-home]: ./v5/at_home/index.html
//! [v5-scanlation-group]: ./v5/scanlation_group/index.html
//! [v5-search]: ./v5/search/index.html
//! [v5-user]: ./v5/user/index.html

#![forbid(unsafe_code)]

pub mod common;
pub mod constants;
#[macro_use]
mod http_client;
pub mod types;
pub mod v5;

pub use constants::*;
pub use http_client::HttpClient;
pub use v5::error::Error;
pub use v5::MangaDexClient;
pub use v5::Result;
