// TOKENIZATION TESTS
use crate::tokenization;

#[test]
fn tokenize_alphabet() {
    let letters = vec!["h", "e", "l", "l", "o"];
    let mut letter_strings: Vec<String> = Vec::new();
    for letter in letters {letter_strings.push(String::from(letter))}
    assert_eq!(tokenization::tokenize_alphabet(vec![String::from("hello")]), vec![letter_strings]);
}

#[test]
fn tokenize_spaces() {
    let tokens = vec!["hello", "how", "are", "you"];
    let mut token_strings: Vec<String> = Vec::new();
    for token in tokens {token_strings.push(String::from(token))}
    assert_eq!(tokenization::tokenize_spaces(vec![String::from("hello how are you")]), vec![token_strings]);
}

#[test]
fn tokenize_bpe() {
    let tokens: Vec<String> = vec!["hello", ",", " ", "how", " ", "are", " ", "you"].iter().map(|str| {str.to_string()}).collect();
    assert_eq!(tokenization::tokenize_bpe(vec![String::from("hello, how are you")], None), vec![tokens.clone()]);
    let tokenizer = tokenization::load_bpe_tokenizer();
    assert_eq!(tokenization::tokenize_bpe(vec![String::from("hello, how are you")], Some(&tokenizer)), vec![tokens]);
}

#[test]
fn tokenize_wordpiece() {
    let tokens: Vec<String> = vec!["hello", ",", "how", "are", "you"].iter().map(|str| {str.to_string()}).collect();
    assert_eq!(tokenization::tokenize_wordpiece(vec![String::from("hello, how are you")], None), vec![tokens.clone()]);
    let tokenizer = tokenization::load_wordpiece_tokenizer();
    assert_eq!(tokenization::tokenize_wordpiece(vec![String::from("hello, how are you")], Some(&tokenizer)), vec![tokens]);
}

#[test]
fn untokenize_bpe() {
    let sentence = vec!["hello, how are you?".to_string()];
    let tokens = tokenization::tokenize_bpe(sentence.clone(), None);
    assert_eq!(tokenization::untokenize_bpe(tokens), sentence);
}

#[test]
fn untokenize_wordpiece() {
    let sentence = vec!["hello, how are you?".to_string()];
    let tokens = tokenization::tokenize_wordpiece(sentence.clone(), None);
    assert_eq!(tokenization::untokenize_wordpiece(tokens), sentence);
}

#[test]
fn untokenize_alphabet() {
    let sentence = vec!["hello, how are you?".to_string()];
    let tokens = tokenization::tokenize_alphabet(sentence.clone());
    assert_eq!(tokenization::untokenize_alphabet(tokens), sentence);
}

#[test]
fn untokenize_spaces() {
    let sentence = vec!["hello, how are you?".to_string()];
    let tokens = tokenization::tokenize_spaces(sentence.clone());
    assert_eq!(tokenization::untokenize_spaces(tokens), sentence);
}