use crate::vocab;
#[test]
fn creating_vocab() {
    let _new_vocab = vocab::Vocab::new();
    let _bpe_vocab = vocab::load_bpe_vocab();
}

#[test]
fn adding_tokens() {
    let mut new_vocab = vocab::Vocab::new();
    // Single token
    new_vocab.add_token(String::from("hello"));
    assert_eq!(new_vocab.num_tokens, 5); // 4 initial tokens, plus added one
    
    // Many tokens
    new_vocab.add_tokens(vec![String::from("hi"), String::from("howdy"), String::from("how goes")]);
    assert_eq!(new_vocab.num_tokens, 8); // 4 initial tokens plus 4 added
}

#[test]
fn removing_tokens() {
    let mut new_vocab = vocab::Vocab::new();
    new_vocab.add_tokens(vec![String::from("hi"), String::from("howdy"), String::from("how goes")]);
    // Single token
    new_vocab.remove_token(String::from("hi"));
    assert_eq!(new_vocab.num_tokens, 6);
    
    // Many tokens
    new_vocab.remove_tokens(vec![String::from("howdy"), String::from("how goes")]);
    assert_eq!(new_vocab.num_tokens, 4);
}

#[test]
fn indexes_from_tokens_bpe() {
    let bpe_vocab = vocab::load_bpe_vocab();
    let tokens = ["Hello", ",", " ", "how", " ", "are", " ", "you", "?"];
    let mut tokens_vec: Vec<String> = Vec::new();
    for token in tokens.iter(){
        tokens_vec.push(String::from(*token));
    }
    let indexes = bpe_vocab.indexes_from_tokens(&tokens_vec);
    assert_eq!(indexes.unwrap(), vec![23858 as u32, 37861, 4, 4786, 4, 290, 4, 3258, 22092]);
}

#[test]
fn indexes_from_tokens_wordpiece() {
    let wordpiece_vocab = vocab::load_wordpiece_vocab();
    let tokens = ["hello", ",", "how", "are", "you", "?"];
    let mut tokens_vec: Vec<String> = Vec::new();
    for token in tokens.iter(){
        tokens_vec.push(String::from(*token));
    }
    let indexes = wordpiece_vocab.indexes_from_tokens(&tokens_vec);
    assert_eq!(indexes.unwrap(), vec![7596 as u32, 1014, 2133, 2028, 2021, 1033]);
}

#[test]
fn tokens_from_indexes_bpe() {
    let bpe_vocab = vocab::load_bpe_vocab();
    let tokens = ["Hello", ",", " ", "how", " ", "are", " ", "you", "?"];
    let mut tokens_vec: Vec<String> = Vec::new();
    for token in tokens.iter(){
        tokens_vec.push(String::from(*token));
    }
    let tokens = bpe_vocab.tokens_from_indexes(&vec![23858 as u32, 37861, 4, 4786, 4, 290, 4, 3258, 22092]);
    assert_eq!(tokens.unwrap(), tokens_vec);
}

#[test]
fn tokens_from_indexes_wordpiece() {
    let wordpiece_vocab = vocab::load_wordpiece_vocab();
    let tokens = ["hello", ",", "how", "are", "you", "?"];
    let mut tokens_vec: Vec<String> = Vec::new();
    for token in tokens.iter(){
        tokens_vec.push(String::from(*token));
    }
    let tokens = wordpiece_vocab.tokens_from_indexes(&vec![7596 as u32, 1014, 2133, 2028, 2021, 1033]);
    assert_eq!(tokens.unwrap(), tokens_vec);
}

#[test]
fn batch_indexes_from_tokens() {
    let bpe_vocab = vocab::load_bpe_vocab();
    let tokens = ["Hello", ",", " ", "how", " ", "are", " ", "you", "?"];
    let mut tokens_vec: Vec<Vec<String>> = vec![Vec::new()];
    for token in tokens.iter(){
        tokens_vec[0].push(String::from(*token));
    }
    let indexes = bpe_vocab.batch_indexes_from_tokens(&tokens_vec);
    assert_eq!(indexes.unwrap()[0], vec![23858 as u32, 37861, 4, 4786, 4, 290, 4, 3258, 22092]);
}

#[test]
fn batch_tokens_from_indexes() {
    let bpe_vocab = vocab::load_bpe_vocab();
    let tokens = ["Hello", ",", " ", "how", " ", "are", " ", "you", "?"];
    let mut tokens_vec: Vec<String> = Vec::new();
    for token in tokens.iter(){
        tokens_vec.push(String::from(*token));
    }
    let tokens = bpe_vocab.batch_tokens_from_indexes(&vec![vec![23858 as u32, 37861, 4, 4786, 4, 290, 4, 3258, 22092]]);
    assert_eq!(tokens.unwrap()[0], tokens_vec);
}