//! Transaction IDs

use crate::{Error, Result};
use rand::{
    distributions::{Distribution, Standard},
    Rng,
};
use serde::{
    de::{self, Visitor},
    Deserialize, Deserializer, Serialize, Serializer,
};
use std::{
    fmt::{self, Debug, Display, Formatter},
    result::Result as StdResult,
};

#[derive(Copy, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct TxId {
    bytes: [u8; MAX_TXID_LEN],
    len: usize,
}

pub const MAX_TXID_LEN: usize = 16;

impl Debug for TxId {
    fn fmt(&self, fmt: &mut Formatter) -> fmt::Result {
        <Self as Display>::fmt(self, fmt)
    }
}

impl Display for TxId {
    fn fmt(&self, fmt: &mut Formatter) -> fmt::Result {
        write!(fmt, "{}", hex::encode(self.as_ref()))
    }
}

impl AsRef<[u8]> for TxId {
    fn as_ref(&self) -> &[u8] {
        &self.bytes[..self.len]
    }
}

impl From<u32> for TxId {
    fn from(id: u32) -> Self {
        let mut bytes = [0u8; MAX_TXID_LEN];
        bytes[..4].copy_from_slice(&id.to_be_bytes());
        Self { bytes, len: 4 }
    }
}

impl From<[u8; MAX_TXID_LEN]> for TxId {
    fn from(bytes: [u8; MAX_TXID_LEN]) -> Self {
        Self {
            bytes,
            len: MAX_TXID_LEN,
        }
    }
}

impl<'a> TryFrom<&'a [u8]> for TxId {
    type Error = Error;
    fn try_from(slice: &'a [u8]) -> Result<Self> {
        if slice.is_empty() || slice.len() > MAX_TXID_LEN {
            return Err(Error::ParseFailure("invalid transaction ID length".into()));
        }
        let len = slice.len();
        let mut bytes = [0u8; MAX_TXID_LEN];
        bytes[..len].copy_from_slice(slice);
        Ok(Self { bytes, len })
    }
}

impl Distribution<TxId> for Standard {
    fn sample<R>(&self, rng: &mut R) -> TxId
    where
        R: Rng + ?Sized,
    {
        <TxId as From<u32>>::from(Standard.sample(rng))
    }
}

impl Serialize for TxId {
    fn serialize<S>(&self, ser: S) -> StdResult<S::Ok, S::Error>
    where
        S: Serializer,
    {
        ser.serialize_bytes(self.as_ref())
    }
}

impl<'de> Deserialize<'de> for TxId {
    fn deserialize<D>(de: D) -> std::result::Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        de.deserialize_bytes(TxIdVisitor)
    }
}

struct TxIdVisitor;

impl<'de> Visitor<'de> for TxIdVisitor {
    type Value = TxId;

    fn expecting(&self, f: &mut Formatter) -> fmt::Result {
        write!(f, "byte array of length <= {}", MAX_TXID_LEN)
    }

    fn visit_bytes<E>(self, v: &[u8]) -> StdResult<Self::Value, E>
    where
        E: de::Error,
    {
        if v.len() > MAX_TXID_LEN {
            return Err(de::Error::invalid_length(v.len(), &"<= 16"));
        }
        Ok(TxId::try_from(v).expect("invalid length"))
    }

    fn visit_borrowed_bytes<E>(self, v: &'de [u8]) -> StdResult<Self::Value, E>
    where
        E: de::Error,
    {
        self.visit_bytes(v)
    }

    fn visit_byte_buf<E>(self, v: Vec<u8>) -> StdResult<Self::Value, E>
    where
        E: de::Error,
    {
        self.visit_bytes(&v)
    }

    fn visit_str<E>(self, s: &str) -> StdResult<Self::Value, E>
    where
        E: de::Error,
    {
        self.visit_bytes(s.as_bytes())
    }

    fn visit_borrowed_str<E>(self, s: &'de str) -> StdResult<Self::Value, E>
    where
        E: de::Error,
    {
        self.visit_str(s)
    }

    fn visit_string<E>(self, s: String) -> StdResult<Self::Value, E>
    where
        E: de::Error,
    {
        self.visit_str(&s)
    }
}
