//! Transport abstraction

use crate::{Error, Result};
use async_std::net::UdpSocket;
use std::{
    future::Future,
    net::{SocketAddr, SocketAddrV4},
    pin::Pin,
};

pub type LocalBoxFuture<'a, T> = Pin<Box<dyn Future<Output = T> + 'a>>;

/// DHT transport abstraction
pub trait Transport {
    /// Attempt to send a datagram
    fn send<'a>(
        &'a mut self,
        buf: &'a [u8],
        addr: SocketAddrV4,
    ) -> LocalBoxFuture<'a, Result<usize>>;

    /// Attempt to receive a datagram
    fn receive<'a>(
        &'a mut self,
        buf: &'a mut [u8],
    ) -> LocalBoxFuture<'a, Result<(usize, SocketAddrV4)>>;
}

impl<T> Transport for &mut T
where
    T: Transport + Unpin + ?Sized,
{
    fn send<'a>(
        &'a mut self,
        buf: &'a [u8],
        addr: SocketAddrV4,
    ) -> LocalBoxFuture<'a, Result<usize>> {
        (&mut **self).send(buf, addr)
    }

    fn receive<'a>(
        &'a mut self,
        buf: &'a mut [u8],
    ) -> LocalBoxFuture<'a, Result<(usize, SocketAddrV4)>> {
        (&mut **self).receive(buf)
    }
}

impl Transport for UdpSocket {
    fn send<'a>(
        &'a mut self,
        buf: &'a [u8],
        addr: SocketAddrV4,
    ) -> LocalBoxFuture<'a, Result<usize>> {
        Box::pin(async move {
            let n = self.send_to(buf, addr).await?;
            Ok(n)
        })
    }

    fn receive<'a>(
        &'a mut self,
        buf: &'a mut [u8],
    ) -> LocalBoxFuture<'a, Result<(usize, SocketAddrV4)>> {
        Box::pin(async move {
            let (n, addr) = self.recv_from(buf).await?;
            match addr {
                SocketAddr::V4(addr) => Ok((n, addr)),
                _ => Err(Error::ReceiveFailure),
            }
        })
    }
}
