//! Write tokens

use crate::{Error, Result};
use rand::{
    distributions::{Distribution, Standard},
    Rng,
};
use serde::{Deserialize, Serialize};
use std::fmt::{self, Debug, Display, Formatter};

#[derive(Clone, PartialEq, Eq, PartialOrd, Ord, Serialize, Deserialize)]
#[serde(into = "TokenSerial", try_from = "TokenSerial")]
pub struct Token(Vec<u8>);

#[derive(Serialize, Deserialize)]
#[serde(transparent)]
pub struct TokenSerial(#[serde(with = "serde_bytes")] Vec<u8>);

pub const MAX_TOKEN_LEN: usize = 32;

impl Debug for Token {
    fn fmt(&self, fmt: &mut Formatter) -> fmt::Result {
        write!(fmt, "{}", self)
    }
}

impl Display for Token {
    fn fmt(&self, fmt: &mut Formatter) -> fmt::Result {
        write!(fmt, "{}", hex::encode(&self.0))
    }
}

impl AsRef<[u8]> for Token {
    fn as_ref(&self) -> &[u8] {
        &self.0
    }
}

impl TryFrom<Vec<u8>> for Token {
    type Error = Error;
    fn try_from(vec: Vec<u8>) -> Result<Self> {
        if vec.is_empty() || vec.len() > MAX_TOKEN_LEN {
            return Err(Error::ParseFailure("invalid token length".into()));
        }
        Ok(Self(vec))
    }
}

impl From<u64> for Token {
    fn from(u: u64) -> Self {
        Self(u.to_be_bytes().to_vec())
    }
}

impl TryFrom<TokenSerial> for Token {
    type Error = Error;
    fn try_from(token: TokenSerial) -> Result<Self> {
        Self::try_from(token.0)
    }
}

impl From<Token> for TokenSerial {
    fn from(token: Token) -> Self {
        Self(token.0)
    }
}

impl Distribution<Token> for Standard {
    fn sample<R>(&self, rng: &mut R) -> Token
    where
        R: Rng + ?Sized,
    {
        let x: [u8; 8] = Standard.sample(rng);
        Token(x.to_vec())
    }
}
