//! Testing network

use crate::{
    message::Message,
    transport::{LocalBoxFuture, Transport},
    Error, Result,
};
use futures::{
    channel::mpsc::{unbounded, UnboundedReceiver, UnboundedSender},
    sink::SinkExt,
    stream::StreamExt,
};
use rand::random;
use serde_bencode::from_bytes as bdecode;
use std::{
    cmp::min as at_most,
    collections::BTreeMap,
    net::{Ipv4Addr, SocketAddrV4},
};

pub struct TestNet {
    receiver: UnboundedReceiver<Packet>,
    peer_sender: UnboundedSender<Packet>,
    senders: BTreeMap<SocketAddrV4, UnboundedSender<Packet>>,
}

impl TestNet {
    pub fn new() -> Self {
        let (peer_sender, receiver) = unbounded();
        Self {
            receiver,
            peer_sender,
            senders: BTreeMap::new(),
        }
    }

    pub fn join(&mut self) -> TestSocket {
        let addr = SocketAddrV4::new(
            Ipv4Addr::new(random(), random(), random(), random()),
            self.senders.len() as u16 + 1,
        );
        let (peer, sender) = TestSocket::new(addr, self.peer_sender.clone());
        self.senders.insert(addr, sender);
        peer
    }

    pub async fn run(mut self) -> Result<()> {
        loop {
            while let Some(packet) = self.receiver.next().await {
                if let Some(sender) = self.senders.get_mut(&packet.destination) {
                    let _ = sender.send(packet).await;
                }
            }
        }
    }
}

pub struct TestSocket {
    addr: SocketAddrV4,
    sender: UnboundedSender<Packet>,
    receiver: UnboundedReceiver<Packet>,
}

impl TestSocket {
    fn new(addr: SocketAddrV4, sender: UnboundedSender<Packet>) -> (Self, UnboundedSender<Packet>) {
        let (my_sender, receiver) = unbounded();
        (
            Self {
                addr,
                sender,
                receiver,
            },
            my_sender,
        )
    }

    pub fn address(&self) -> SocketAddrV4 {
        self.addr
    }
}

impl Transport for TestSocket {
    fn send<'a>(
        &'a mut self,
        buf: &'a [u8],
        addr: SocketAddrV4,
    ) -> LocalBoxFuture<'a, Result<usize>> {
        let packet = Packet {
            source: self.addr,
            destination: addr,
            payload: buf.to_vec(),
        };
        let _len = buf.len();
        Box::pin(async move {
            eprintln!(" # # # PACKET {}", packet.to_string());
            self.sender.send(packet).await?;
            Ok(buf.len())
        })
    }

    fn receive<'a>(
        &'a mut self,
        buf: &'a mut [u8],
    ) -> LocalBoxFuture<'a, Result<(usize, SocketAddrV4)>> {
        Box::pin(async move {
            if let Some(packet) = self.receiver.next().await {
                let n = packet.payload.len();
                let to_read = at_most(buf.len(), n);
                buf[..to_read].copy_from_slice(&packet.payload[..to_read]);
                Ok((to_read, packet.source))
            } else {
                Err(Error::ReceiveFailure)
            }
        })
    }
}

struct Packet {
    source: SocketAddrV4,
    destination: SocketAddrV4,
    payload: Vec<u8>,
}

impl Packet {
    fn to_string(&self) -> String {
        let payload = bdecode::<Message>(&self.payload).unwrap();
        format!(
            "[ from {:>21} to {:>21}: {:?} ]",
            self.source, self.destination, payload
        )
    }
}
