//! DHT node

use crate::{
    directory::Directory,
    id::{Id, InfoHash},
    message::{
        ErrorCode, ErrorMessage, ExpectedResponse, Message, Port, QueryMessage, Response,
        ResponseMessage,
    },
    node_info::NodeInfo,
    table::Table,
    timer::Timer,
    token::Token,
    transport::Transport,
    txid::TxId,
    Error, Result,
};
use async_stream::try_stream;
use async_std::channel::{bounded, Receiver as BoundedReceiver, Sender as BoundedSender};
use futures::{
    channel::{
        mpsc::{unbounded, UnboundedReceiver, UnboundedSender},
        oneshot::{channel, Receiver, Sender},
    },
    pending, poll, ready,
    stream::{FuturesUnordered, Stream, StreamExt},
    try_join,
};
use pin_project::pin_project;
use rand::random;
use serde_bencode::{from_bytes as bdecode, to_bytes as bencode};
use std::{
    borrow::Borrow,
    cmp::Ordering,
    collections::{btree_map::Entry, BTreeMap, BTreeSet},
    future::Future,
    io::{self, ErrorKind},
    net::{SocketAddr, SocketAddrV4, ToSocketAddrs},
    pin::Pin,
    result::Result as StdResult,
    task::{
        Context,
        Poll::{self, Pending, Ready},
    },
    time::Duration,
    vec,
};

/// A DHT node
pub struct Node {
    transport: Pin<Box<dyn Transport + Unpin>>,

    routers: Vec<Router>,
    directory: Directory,
    table: Table,

    purge_queries_timer: Timer,
    refresh_buckets_timer: Timer,
    expire_peers_timer: Timer,

    query_handle: QueryHandle,
    query_queue: UnboundedReceiver<QuerySlot>,

    closest_sender: UnboundedSender<(Id, Sender<Vec<NodeInfo>>)>,
    closest_queue: UnboundedReceiver<(Id, Sender<Vec<NodeInfo>>)>,

    peer_queue: UnboundedReceiver<(InfoHash, SocketAddrV4)>,

    bootstrap_notify: BoundedSender<()>,

    pending_replies: Vec<(Message, SocketAddrV4)>,
    pending_queries: BTreeSet<QuerySlot>,
    pending_tokens: BTreeSet<Token>,

    available_tokens: BTreeSet<Token>,
}

impl Node {
    /// Creates a new node over the given transport
    pub fn new<T>(my_id: Id, transport: T) -> (Self, Handle)
    where
        T: Transport + Unpin + 'static,
    {
        let (query_sender, query_queue) = unbounded();
        let (closest_sender, closest_queue) = unbounded();
        let (peer_sender, peer_queue) = unbounded();
        let (bootstrap_notify, bootstrap_done) = bounded(1);
        let query_handle = QueryHandle { query_sender };
        (
            Self {
                transport: Box::pin(transport),
                purge_queries_timer: Timer::start(Self::PURGE_QUERIES_INTERVAL),
                refresh_buckets_timer: Timer::start(Self::REFRESH_BUCKETS_INTERVAL),
                expire_peers_timer: Timer::start(Self::EXPIRE_PEERS_INTERVAL),
                routers: vec![],
                directory: Directory::new(my_id, query_handle.clone()),
                table: Table::new(),
                query_handle: query_handle.clone(),
                query_queue,
                closest_sender: closest_sender.clone(),
                closest_queue,
                peer_queue,
                bootstrap_notify,
                pending_replies: Vec::new(),
                pending_queries: BTreeSet::new(),
                pending_tokens: BTreeSet::new(),
                available_tokens: BTreeSet::new(),
            },
            Handle {
                my_id,
                query_handle,
                closest_sender,
                peer_sender,
                bootstrap_signal: Some(bootstrap_done),
            },
        )
    }

    /// Adds a router to be used in the bootstrap process
    pub fn add_router(&mut self, router: Router) {
        self.routers.push(router);
    }

    /// Adds a known node to the routing table
    pub fn add_node(&mut self, addr: SocketAddrV4) {
        self.add_router(Router::Custom(addr));
    }

    /// Runs the node
    pub async fn run(mut self) -> Result<()> {
        self.preallocate_tokens();
        try_join!(self.bootstrap(), self.do_run())?;
        Ok(())
    }

    async fn do_run(&mut self) -> Result<()> {
        loop {
            self.dispatch_incoming().await?;
            self.dispatch_outgoing().await?;
            self.perform_maintenance().await?;
            pending!();
        }
    }

    fn bootstrap(&self) -> impl Future<Output = Result<()>> {
        let routers = self.routers.clone();
        let query_handle = self.query_handle.clone();
        let bootstrap_notify = self.bootstrap_notify.clone();
        let find_neighbors = self.find_neighbors();

        async move {
            let boostrap_timer = Timer::start(Self::BOOTSTRAP_TIME);
            let mut futs = FuturesUnordered::new();
            for router in routers.iter() {
                let addr = router.socket_addr()?;
                futs.push(query_handle.ping(addr)?);
            }
            println!("Bootstrappin'");
            loop {
                match poll!(futs.next()) {
                    Ready(None) => break,
                    Pending => {
                        if boostrap_timer.is_expired() {
                            break;
                        } else {
                            pending!()
                        }
                    }
                    _ => {}
                }
            }

            find_neighbors.await?;
            let _ = bootstrap_notify.send(()).await;
            Ok(())
        }
    }

    async fn dispatch_incoming(&mut self) -> Result<()> {
        let my_id = self.directory.id();
        let mut buf = [0u8; Self::MAX_DATAGRAM_LEN];
        while let Ready(res) = poll!(self.transport.receive(&mut buf)) {
            let (n, address) = res?;
            let buf = &buf[..n];
            if let Ok(message) = bdecode::<Message>(buf) {
                match message {
                    Message::Query { txid, message } => {
                        self.directory.mark_query(message.id());
                        match message {
                            QueryMessage::Ping { id } => {
                                self.found_node(id, address);
                                let message = Message::Response {
                                    txid,
                                    message: Response::Pong { id: my_id }.into(),
                                };
                                self.pending_replies.push((message, address));
                            }
                            QueryMessage::FindNode { id, target } => {
                                self.found_node(id, address);
                                let nodes = self.directory.closest_nodes(target);
                                let message = Message::Response {
                                    txid,
                                    message: Response::ClosestNodes { id: my_id, nodes }.into(),
                                };
                                self.pending_replies.push((message, address));
                            }
                            QueryMessage::GetPeers { id, info_hash } => {
                                self.found_node(id, address);
                                let token = self.get_token();
                                let message = match self.table.get_addresses(&info_hash) {
                                    Some(peers) => Message::Response {
                                        txid,
                                        message: Response::KnownPeers {
                                            id: my_id,
                                            token: token.clone(),
                                            peers,
                                        }
                                        .into(),
                                    },
                                    None => {
                                        let nodes = self.directory.closest_nodes(info_hash);
                                        Message::Response {
                                            txid,
                                            message: Response::NoKnownPeers {
                                                id: my_id,
                                                token: token.clone(),
                                                nodes,
                                            }
                                            .into(),
                                        }
                                    }
                                };
                                self.pending_replies.push((message, address));
                                self.pending_tokens.insert(token);
                            }
                            QueryMessage::AnnouncePeer {
                                id,
                                token,
                                info_hash,
                                port,
                            } => {
                                self.found_node(id, address);
                                let message = match self.pending_tokens.take(&token) {
                                    Some(_) => {
                                        let port = port.unwrap_or(address.port());
                                        let address = SocketAddrV4::new(*address.ip(), port);
                                        self.table.insert(info_hash, address);
                                        Message::Response {
                                            txid,
                                            message: Response::Announced { id: my_id }.into(),
                                        }
                                    }
                                    None => Message::Error {
                                        txid,
                                        message: ErrorMessage::new(
                                            ErrorCode::ProtocolError,
                                            "bad token",
                                        ),
                                    },
                                };
                                self.pending_replies.push((message, address));
                            }
                        }
                    }
                    Message::Response { txid, message } => {
                        if let Some(slot) = self.pending_queries.take(&txid) {
                            self.found_node(message.id(), address);
                            self.directory.mark_response(message.id());
                            slot.answer(message);
                        }
                    }
                    Message::Error { txid, message } => {
                        if let Some(slot) = self.pending_queries.take(&txid) {
                            slot.fail(message);
                        }
                    }
                }
            }
        }
        Ok(())
    }

    async fn dispatch_outgoing(&mut self) -> Result<()> {
        while let Ready(Some(mut slot)) = poll!(self.query_queue.next()) {
            slot.message.set_id(self.directory.id());
            let buf = bencode(&slot.message).expect("invalid message");
            self.transport.send(&buf, slot.address).await?;
            if let Some(id) = self.directory.id_of(slot.address) {
                self.directory.mark_ping(id);
            }
            self.pending_queries.insert(slot);
        }
        for (mut message, address) in self.pending_replies.drain(..) {
            message.set_id(self.directory.id());
            let buf = bencode(&message).expect("invalid message");
            self.transport.send(&buf, address).await?;
        }
        Ok(())
    }

    async fn perform_maintenance(&mut self) -> Result<()> {
        while let Ready(Some((id, addr))) = poll!(self.peer_queue.next()) {
            self.table.insert(id, addr);
        }
        while let Ready(Some((id, reply))) = poll!(self.closest_queue.next()) {
            let closest = self.directory.closest_nodes(id);
            let _ = reply.send(closest);
        }
        if self.purge_queries_timer.is_expired() {
            let mut retried = vec![];
            let expired = self.pending_queries.drain_filter(QuerySlot::is_expired);
            for mut slot in expired {
                if slot.exhausted_retries() {
                    slot.timeout();
                } else {
                    slot.mark_retry();
                    retried.push(slot);
                }
            }
            self.pending_queries.extend(retried);
            self.purge_queries_timer.restart();
        }
        if self.refresh_buckets_timer.is_expired() {
            self.directory.refresh();
            self.refresh_buckets_timer.restart();
        }
        if self.expire_peers_timer.is_expired() {
            self.table.expire();
            self.expire_peers_timer.restart();
        }
        if self.available_tokens.is_empty() {
            self.preallocate_tokens();
        }
        Ok(())
    }

    fn preallocate_tokens(&mut self) {
        while self.available_tokens.len() < Self::TOKEN_BATCH_SIZE {
            self.available_tokens.insert(random());
        }
    }

    fn find_neighbors(&self) -> impl Future<Output = Result<()>> {
        let my_id = self.directory.id();
        let query_handle = self.query_handle.clone();
        let closest_sender = self.closest_sender.clone();

        async move {
            let mut buf = [0u8; Id::LEN];
            let last_idx = Id::LEN - 1;
            buf[..last_idx].copy_from_slice(&my_id.as_ref()[..last_idx]);
            buf[last_idx] = random();
            let target = Id::from(buf);
            let (sender, receiver) = channel();
            closest_sender.unbounded_send((target, sender))?;
            let nodes = receiver.await.expect("oneshot cancelled");

            let mut pending: BTreeMap<_, _> = nodes
                .into_iter()
                .map(|node| (node, SearchStatus::Waiting))
                .collect();
            let mut queries = pending
                .iter()
                .map(|(node, _)| query_handle.find_node(target, node.address()))
                .collect::<Result<FuturesUnordered<_>>>()?;

            let mut queries_done = false;
            while !queries_done && !fully_resolved(&pending) {
                match queries.next().await {
                    Some(Ok(Response::ClosestNodes { id, nodes })) => {
                        resolve_node(&mut pending, id, None);
                        for node in nodes {
                            if node.id() == my_id || !insert_search_node(&mut pending, node) {
                                continue;
                            }
                            if let Ok(query) = query_handle.find_node(target, node.address()) {
                                queries.push(query);
                            }
                        }
                    }
                    None => queries_done = true,
                    _ => {}
                }
            }
            Ok(())
        }
    }

    fn get_token(&mut self) -> Token {
        loop {
            match self.available_tokens.pop_first() {
                Some(token) => {
                    break token;
                }
                None => self.preallocate_tokens(),
            }
        }
    }

    fn found_node(&mut self, id: Id, address: SocketAddrV4) {
        self.directory.add_node(NodeInfo::new(id, address));
    }

    #[cfg(test)]
    fn get_query_handle(&self) -> QueryHandle {
        self.query_handle.clone()
    }

    const MAX_DATAGRAM_LEN: usize = 1024;
    const TOKEN_BATCH_SIZE: usize = 64;
    const BOOTSTRAP_TIME: Duration = Duration::from_secs(15 * 60);
    const PURGE_QUERIES_INTERVAL: Duration = Duration::from_secs(10);
    const REFRESH_BUCKETS_INTERVAL: Duration = Duration::from_secs(60);
    const EXPIRE_PEERS_INTERVAL: Duration = Duration::from_secs(60);
    const NEIGHBORHOOD_SCAN_INTERVAL: Duration = Duration::from_secs(37 * 60);
}

/// Known DHT routers for bootstrapping
#[derive(Copy, Clone, Debug)]
pub enum Router {
    /// Bootstrap server maintained by uTorrent
    UTorrent,
    /// Bootstrap server maintained by BitTorrent
    BitTorrent,
    /// Bootstrap server maintained by BitComet
    BitComet,
    /// Bootstrap server maintained by Transmission
    Transmission,
    /// Custom bootstrap server
    Custom(SocketAddrV4),
}

impl Router {
    fn socket_addr(&self) -> Result<SocketAddrV4> {
        match self {
            Router::UTorrent => Self::get_socket_addr(Self::UTORRENT_ADDRESS),
            Router::BitTorrent => Self::get_socket_addr(Self::BITTORRENT_ADDRESS),
            Router::BitComet => Self::get_socket_addr(Self::BITCOMET_ADDRESS),
            Router::Transmission => Self::get_socket_addr(Self::TRANSMISSION_ADDRESS),
            Router::Custom(addr) => Ok(*addr),
        }
    }

    fn get_socket_addr(addr: (&str, u16)) -> Result<SocketAddrV4> {
        let make_error = || io::Error::new(ErrorKind::Other, "no socket addresses found");
        let mut socket_addrs = addr.to_socket_addrs()?;
        let socket_addr = socket_addrs.next().ok_or_else(make_error)?;
        Ok(Self::map_ipv4(socket_addr).ok_or_else(make_error)?)
    }

    fn map_ipv4(addr: SocketAddr) -> Option<SocketAddrV4> {
        match addr {
            SocketAddr::V4(n) => Some(n),
            _ => None,
        }
    }

    const UTORRENT_ADDRESS: (&'static str, u16) = ("router.utorrent.com", 6881);
    const BITTORRENT_ADDRESS: (&'static str, u16) = ("router.bittorrent.com", 6881);
    const BITCOMET_ADDRESS: (&'static str, u16) = ("router.bitcomet.com", 6881);
    const TRANSMISSION_ADDRESS: (&'static str, u16) = ("dht.transmissionbt.com", 6881);
}

/// A handle for interacting with the DHT
#[derive(Clone)]
pub struct Handle {
    my_id: Id,
    query_handle: QueryHandle,
    closest_sender: UnboundedSender<(Id, Sender<Vec<NodeInfo>>)>,
    peer_sender: UnboundedSender<(InfoHash, SocketAddrV4)>,
    bootstrap_signal: Option<BoundedReceiver<()>>,
}

impl Handle {
    /// Waits for the node to finish bootstrapping
    pub async fn wait_for_bootstrap(&mut self) -> Result<()> {
        if let Some(signal) = &self.bootstrap_signal {
            signal.recv().await?;
            self.bootstrap_signal.take();
        }
        Ok(())
    }

    /// Performs a search for peers downloading a torrent, optionally announcing yourself
    pub fn search(
        &self,
        info_hash: InfoHash,
        announce: Announce,
    ) -> impl Stream<Item = Result<SocketAddrV4>> + '_ {
        let closest_sender = self.closest_sender.clone();
        let query_handle = self.query_handle.clone();
        let my_id = self.my_id;

        try_stream! {
            let search_timer = Timer::start(Self::SEARCH_TIME);
            let (sender, receiver) = channel();
            closest_sender.unbounded_send((info_hash, sender))?;
            let nodes = receiver.await.expect("oneshot cancelled");
            let mut found_peers = BTreeSet::new();
            let mut pending: BTreeMap<_, _> = nodes
                .into_iter()
                .map(|node| (node, SearchStatus::Waiting))
                .collect();
            let mut queries = pending
                .iter()
                .map(|(node, _)| query_handle.get_peers(info_hash, node.address()))
                .collect::<Result<FuturesUnordered<_>>>()?;

            let mut queries_done = false;
            while !queries_done && !fully_resolved(&pending) && !search_timer.is_expired() {
                match queries.next().await {
                    Some(result) => match result {
                        Ok(Response::KnownPeers { id, peers, token }) => {
                            resolve_node(&mut pending, id, Some(token));
                            for peer in peers {
                                if found_peers.insert(peer) {
                                    yield peer;
                                }
                            }
                        }
                        Ok(Response::NoKnownPeers { id, nodes, token }) => {
                            resolve_node(&mut pending, id, Some(token));
                            for node in nodes {
                                if node.id() == my_id || !insert_search_node(&mut pending, node) {
                                    continue;
                                }
                                if let Ok(query) = query_handle.get_peers(info_hash, node.address())
                                {
                                    queries.push(query);
                                }
                            }
                        }
                        _ => {}
                    },
                    None => queries_done = true,
                }
            }

            let announce_port = match announce {
                Announce::SamePort => Some(Port::Implicit),
                Announce::OnPort(port) => Some(Port::Explicit(port)),
                Announce::No => None,
            };
            if let Some(port) = announce_port {
                for (node, status) in pending {
                    if let SearchStatus::ResolvedWith(token) = status {
                        let _ = query_handle.announce_peer(token, info_hash, port, node.address());
                    }
                }
            }
        }
    }

    /// Adds a known peer to the table
    pub fn add_peer(&self, info_hash: InfoHash, addr: SocketAddrV4) -> Result<()> {
        Ok(self.peer_sender.unbounded_send((info_hash, addr))?)
    }

    const SEARCH_TIME: Duration = Duration::from_secs(15 * 60);
}

/// Announcement setting
#[derive(Copy, Clone, PartialEq, Eq)]
pub enum Announce {
    /// Do not announce
    No,
    /// Announce on the same port as DHT
    SamePort,
    /// Announce on a specific port
    OnPort(u16),
}

#[derive(Clone)]
pub enum Event {
    BootstrapStarted,
    BootstrapCompleted,
}

#[derive(Clone, Debug, PartialEq, Eq)]
enum SearchStatus {
    Waiting,
    Resolved,
    ResolvedWith(Token),
}

fn insert_search_node(nodes: &mut BTreeMap<NodeInfo, SearchStatus>, node: NodeInfo) -> bool {
    if nodes.len() > 2 * Directory::BUCKET_SIZE && nodes.last_key_value().unwrap().0 < &node {
        return false;
    }
    let entry = nodes.entry(node);
    if let Entry::Occupied(o) = &entry {
        if matches!(
            o.get(),
            SearchStatus::Resolved | SearchStatus::ResolvedWith(_)
        ) {
            return false;
        }
    }
    entry.or_insert(SearchStatus::Waiting);
    while nodes.len() > 2 * Directory::BUCKET_SIZE {
        nodes.pop_last(); // TODO return false if popped?
    }

    nodes.insert(node, SearchStatus::Waiting);
    true
}

fn resolve_node(nodes: &mut BTreeMap<NodeInfo, SearchStatus>, id: Id, token: Option<Token>) {
    if let Some(s) = nodes.get_mut(&id) {
        *s = match token {
            Some(token) => SearchStatus::ResolvedWith(token),
            None => SearchStatus::Resolved,
        };
    }
}

fn fully_resolved(nodes: &BTreeMap<NodeInfo, SearchStatus>) -> bool {
    nodes
        .iter()
        .take(Directory::BUCKET_SIZE)
        .all(|(_, replied)| {
            matches!(
                replied,
                SearchStatus::Resolved | SearchStatus::ResolvedWith(_)
            )
        })
}

/// A handle for performing queries
#[derive(Clone)]
pub struct QueryHandle {
    query_sender: UnboundedSender<QuerySlot>,
}

impl QueryHandle {
    /// Performs a ping query
    pub fn ping(&self, address: SocketAddrV4) -> Result<ReplyHandle> {
        self.query(QueryMessage::ping(), address)
    }

    /// Performs a find_node query
    pub fn find_node(&self, target: Id, address: SocketAddrV4) -> Result<ReplyHandle> {
        self.query(QueryMessage::find_node(target), address)
    }

    /// Performs a get_peers query
    pub fn get_peers(&self, info_hash: Id, address: SocketAddrV4) -> Result<ReplyHandle> {
        self.query(QueryMessage::get_peers(info_hash), address)
    }

    /// Performs an announce_peer query
    pub fn announce_peer(
        &self,
        token: Token,
        info_hash: Id,
        port: Port,
        address: SocketAddrV4,
    ) -> Result<ReplyHandle> {
        self.query(QueryMessage::announce_peer(token, info_hash, port), address)
    }

    /// Performs a query and returns a handle to receive the reply
    fn query(&self, message: QueryMessage, address: SocketAddrV4) -> Result<ReplyHandle> {
        let (slot, reply_receiver) = QuerySlot::new(message, address);
        self.query_sender.unbounded_send(slot)?;
        Ok(ReplyHandle { reply_receiver })
    }
}

/// A handle for receiving replies to queries
#[pin_project]
pub struct ReplyHandle {
    #[pin]
    reply_receiver: Receiver<QueryResult>,
}

impl Future for ReplyHandle {
    type Output = Result<Response>;

    fn poll(self: Pin<&mut Self>, cx: &mut Context) -> Poll<Result<Response>> {
        let this = self.project();
        Ready(Ok(
            ready!(this.reply_receiver.poll(cx)).expect("cancelled reply")?
        ))
    }
}

type QueryResult = StdResult<Response, QueryError>;

struct QuerySlot {
    txid: TxId,
    message: Message,
    expected_response: ExpectedResponse,
    address: SocketAddrV4,
    reply: Sender<QueryResult>,
    timer: Timer,
    attempts: u64,
}

impl QuerySlot {
    pub fn new(message: QueryMessage, address: SocketAddrV4) -> (Self, Receiver<QueryResult>) {
        let txid = random();
        let (sender, receiver) = channel();
        let expected_response = message.expected_response();
        (
            Self {
                txid,
                message: Message::Query { txid, message },
                expected_response,
                address,
                reply: sender,
                timer: Timer::start(Self::QUERY_TIMEOUT),
                attempts: 1,
            },
            receiver,
        )
    }

    pub fn is_expired(&self) -> bool {
        self.timer.is_expired()
    }

    pub fn exhausted_retries(&self) -> bool {
        self.attempts >= Self::MAX_ATTEMPTS
    }

    pub fn mark_retry(&mut self) {
        self.attempts += 1;
        self.timer.restart();
    }

    pub fn answer(self, response: ResponseMessage) {
        let parsed = response
            .parse(self.expected_response)
            .map_err(|_| ErrorMessage::protocol("non-matching response").into());
        let _ = self.reply.send(parsed);
    }

    pub fn fail(self, error: ErrorMessage) {
        let _ = self.reply.send(Err(QueryError::Reply(error)));
    }

    pub fn timeout(self) {
        let _ = self.reply.send(Err(QueryError::Timeout));
    }

    const MAX_ATTEMPTS: u64 = 5;
    const QUERY_TIMEOUT: Duration = Duration::from_secs(60);
}

impl PartialEq for QuerySlot {
    fn eq(&self, other: &Self) -> bool {
        self.txid == other.txid
    }
}

impl Eq for QuerySlot {}

impl PartialOrd for QuerySlot {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.txid.partial_cmp(&other.txid)
    }
}

impl Ord for QuerySlot {
    fn cmp(&self, other: &Self) -> Ordering {
        self.txid.cmp(&other.txid)
    }
}

impl Borrow<TxId> for QuerySlot {
    fn borrow(&self) -> &TxId {
        &self.txid
    }
}

#[derive(Clone, Debug)]
pub enum QueryError {
    Timeout,
    Reply(ErrorMessage),
}

impl From<QueryError> for Error {
    fn from(e: QueryError) -> Self {
        match e {
            QueryError::Timeout => Self::Timeout,
            QueryError::Reply(m) => Self::from(m),
        }
    }
}

impl From<ErrorMessage> for QueryError {
    fn from(e: ErrorMessage) -> Self {
        Self::Reply(e)
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use crate::{
        message::{ErrorCode, Port},
        testnet::TestNet,
    };
    use futures::{executor::block_on, future::abortable, join, pin_mut};

    fn make_node(net: &mut TestNet, id: Id) -> (Node, Handle, SocketAddrV4) {
        let sock = net.join();
        let addr = sock.address();
        let (node, handle) = Node::new(id, sock);
        (node, handle, addr)
    }

    fn make_network(n: u8) -> (TestNet, Vec<SocketAddrV4>, impl Future<Output = Result<()>>) {
        let mut net = TestNet::new();
        let mut addrs = vec![];
        let run_futs = FuturesUnordered::new();

        for i in 1..=n {
            let mut buf = random::<[u8; Id::LEN]>();
            buf[Id::LEN - 1] = i;
            let id = Id::from(buf);
            let (mut node, _, addr) = make_node(&mut net, id);
            addrs.push(addr);
            node.add_router(Router::Custom(addrs[0]));
            for j in 2..i {
                if i % j == 0 {
                    node.add_router(Router::Custom(addrs[j as usize - 1]));
                }
            }
            run_futs.push(node.run());
        }
        let run = run_futs.fold(Ok(()), |acc, res| async { acc.and(res) });
        (net, addrs, run)
    }

    #[test]
    fn nodes_bootstrapping_works() {
        let (mut net, addrs, run_others) = make_network(255);
        let id = Id::from(*b"ABCDEFGHIJKLMNOPQRST");
        let (mut node, mut handle, _) = make_node(&mut net, id);
        node.add_router(Router::Custom(addrs[0]));
        let run_net = net.run();
        let run_node = node.run();
        let (run, abort) = abortable(async move { try_join!(run_node, run_net, run_others) });

        let queries = async move {
            handle.wait_for_bootstrap().await.unwrap();

            abort.abort();
            Ok(())
        };

        let _ = block_on(async move { try_join!(run, queries) });
    }

    #[test]
    fn nodes_can_search() {
        let mut net = TestNet::new();
        let sock1 = net.join();
        let sock2 = net.join();
        let sock3 = net.join();
        let sock4 = net.join();
        let addr1 = sock1.address();
        let addr2 = sock2.address();
        let addr3 = sock3.address();
        let addr4 = sock4.address();
        let id1 = random();
        let id2 = random();
        let id3 = random();
        let id4 = random();
        let (node1, handle1) = Node::new(id1, sock1);
        let (node2, _) = Node::new(id2, sock2);
        let (node3, _) = Node::new(id3, sock3);
        let (node4, _) = Node::new(id4, sock4);
        let query1 = node1.get_query_handle();
        let query2 = node2.get_query_handle();
        let query3 = node3.get_query_handle();
        let query4 = node4.get_query_handle();

        let run0 = net.run();
        let run1 = node1.run();
        let run2 = node2.run();
        let run3 = node3.run();
        let run4 = node4.run();

        let (run, abort) = abortable(async move { join!(run0, run1, run2, run3, run4) });

        let replies = async move {
            let r = query1.ping(addr2).unwrap().await.unwrap();
            assert_eq!(r, Response::Pong { id: id2 });

            let r = query2.ping(addr3).unwrap().await.unwrap();
            assert_eq!(r, Response::Pong { id: id3 });

            let r = query3.ping(addr4).unwrap().await.unwrap();
            assert_eq!(r, Response::Pong { id: id4 });

            let r = query4.ping(addr1).unwrap().await.unwrap();
            assert_eq!(r, Response::Pong { id: id1 });

            let r = query1.find_node(id3, addr2).unwrap().await.unwrap();
            let mut expected_nodes = [NodeInfo::new(id3, addr3), NodeInfo::new(id1, addr1)];
            expected_nodes.sort();
            assert!(
                matches!(&r, Response::ClosestNodes { id, nodes } if id == &id2 && {
                    let mut nodes = nodes.clone();
                    nodes.sort();
                    nodes == &expected_nodes
                })
            );

            let r = query1.ping(addr3).unwrap().await.unwrap();
            assert_eq!(r, Response::Pong { id: id3 });

            let info_hash = random();
            let r = query4.get_peers(info_hash, addr2).unwrap().await.unwrap();
            assert!(
                matches!(&r, Response::NoKnownPeers {id, nodes, ..} if id == &id2 && !nodes.is_empty())
            );
            let token = r.token().unwrap();

            let r = query4
                .announce_peer(token.clone(), info_hash, Port::Implicit, addr2)
                .unwrap()
                .await
                .unwrap();
            assert_eq!(r, Response::Announced { id: id2 });

            let r = query3.get_peers(info_hash, addr2).unwrap().await.unwrap();
            let expected_peers = [addr4];
            assert!(
                matches!(&r, Response::KnownPeers { id, peers, .. } if id == &id2 && peers == &expected_peers)
            );

            let e = query3
                .announce_peer(token.clone(), info_hash, Port::Implicit, addr2)
                .unwrap()
                .await
                .unwrap_err();
            assert!(matches!(e, Error::ErrorReply(e) if e.code() == ErrorCode::ProtocolError));
            let token = r.token().unwrap();

            let r = query3
                .announce_peer(token.clone(), info_hash, Port::Implicit, addr2)
                .unwrap()
                .await
                .unwrap();
            assert_eq!(r.id(), id2);

            let r = query4.get_peers(info_hash, addr2).unwrap().await.unwrap();
            let mut expected_peers = [addr3, addr4];
            expected_peers.sort();
            assert!(
                matches!(&r, Response::KnownPeers { id, peers, .. } if id == &id2 && {
                    let mut peers = peers.clone();
                    peers.sort();
                    peers == expected_peers
                })
            );

            {
                let s = handle1.search(info_hash, Announce::No);
                pin_mut!(s);
                let mut v = vec![];
                while let Some(p) = s.next().await {
                    v.push(p.unwrap());
                }
                v.sort();
                let mut expected = [addr3, addr4];
                expected.sort();
                assert_eq!(&v, &expected);
            }

            {
                let s = handle1.search(info_hash, Announce::SamePort);
                pin_mut!(s);
                let mut v = vec![];
                while let Some(p) = s.next().await {
                    v.push(p.unwrap());
                }
                v.sort();
                let mut expected = [addr3, addr4];
                expected.sort();
                assert_eq!(&v, &expected);
            }
            {
                let s = handle1.search(info_hash, Announce::No);
                pin_mut!(s);
                let mut v = vec![];
                while let Some(p) = s.next().await {
                    v.push(p.unwrap());
                }
                v.sort();
                let mut expected = [addr1, addr3, addr4];
                expected.sort();
                assert_eq!(&v, &expected);
            }

            abort.abort();
            Ok(())
        };

        let _ = block_on(async move { try_join!(replies, run) });
    }
}
