//! Wire messages

use crate::{
    id::{Id, InfoHash},
    node_info::NodeInfo,
    serde::{is_false, serde_bool_as_0_or_not},
    token::Token,
    txid::TxId,
    Error, Result,
};
use serde::{Deserialize, Serialize};
use serde_bencode::value::Value as Bencoded;
use std::{
    error::Error as StdError,
    fmt::{self, Debug, Display, Formatter},
    net::SocketAddrV4,
};

#[derive(Clone, Debug, Serialize, Deserialize, PartialEq, Eq)]
#[serde(tag = "y")]
pub enum Message {
    #[serde(rename = "q")]
    Query {
        #[serde(rename = "t")]
        txid: TxId,
        #[serde(flatten)]
        message: QueryMessage,
    },
    #[serde(rename = "r")]
    Response {
        #[serde(rename = "t")]
        txid: TxId,
        #[serde(rename = "r")]
        message: ResponseMessage,
    },
    #[serde(rename = "e")]
    Error {
        #[serde(rename = "t")]
        txid: TxId,
        #[serde(rename = "e")]
        message: ErrorMessage,
    },
}

impl Message {
    pub fn set_id(&mut self, my_id: Id) {
        match self {
            Message::Query { message, .. } => message.set_id(my_id),
            Message::Response { message, .. } => message.set_id(my_id),
            _ => {}
        }
    }

    pub fn id(&self) -> Option<Id> {
        match self {
            Message::Query { message, .. } => Some(message.id()),
            Message::Response { message, .. } => Some(message.id()),
            _ => None,
        }
    }
}

/// A query message
#[derive(Clone, Debug, Serialize, Deserialize, PartialEq, Eq)]
#[serde(into = "QueryMessageSerial", from = "QueryMessageSerial")]
pub enum QueryMessage {
    #[doc(hidden)]
    Ping { id: Id },
    #[doc(hidden)]
    FindNode { id: Id, target: Id },
    #[doc(hidden)]
    GetPeers { id: Id, info_hash: InfoHash },
    #[doc(hidden)]
    AnnouncePeer {
        id: Id,
        token: Token,
        info_hash: InfoHash,
        #[serde(flatten)]
        port: Port,
    },
}

impl QueryMessage {
    /// Creates a new ping query
    pub fn ping() -> Self {
        Self::Ping { id: Id::ZERO }
    }

    /// Creates a new find_node query
    pub fn find_node(target: Id) -> Self {
        Self::FindNode {
            id: Id::ZERO,
            target,
        }
    }

    /// Creates a new get_peers query
    pub fn get_peers(info_hash: InfoHash) -> Self {
        Self::GetPeers {
            id: Id::ZERO,
            info_hash,
        }
    }

    /// Creates a new announce_peer query
    pub fn announce_peer(token: Token, info_hash: InfoHash, port: Port) -> Self {
        Self::AnnouncePeer {
            id: Id::ZERO,
            token,
            info_hash,
            port,
        }
    }

    pub fn id(&self) -> Id {
        match self {
            Self::Ping { id } => *id,
            Self::FindNode { id, .. } => *id,
            Self::GetPeers { id, .. } => *id,
            Self::AnnouncePeer { id, .. } => *id,
        }
    }

    pub fn expected_response(&self) -> ExpectedResponse {
        match self {
            Self::Ping { .. } => ExpectedResponse::Pong,
            Self::FindNode { .. } => ExpectedResponse::ClosestNodes,
            Self::GetPeers { .. } => ExpectedResponse::GotPeers,
            Self::AnnouncePeer { .. } => ExpectedResponse::Announced,
        }
    }

    fn set_id(&mut self, my_id: Id) {
        match self {
            Self::Ping { id } => *id = my_id,
            Self::FindNode { id, .. } => *id = my_id,
            Self::GetPeers { id, .. } => *id = my_id,
            Self::AnnouncePeer { id, .. } => *id = my_id,
        }
    }
}

#[derive(Serialize, Deserialize)]
#[serde(tag = "q")]
pub enum QueryMessageSerial {
    #[serde(rename = "ping")]
    Ping {
        #[serde(rename = "a")]
        args: PingArgs,
    },
    #[serde(rename = "find_node")]
    FindNode {
        #[serde(rename = "a")]
        args: FindNodeArgs,
    },
    #[serde(rename = "get_peers")]
    GetPeers {
        #[serde(rename = "a")]
        args: GetPeersArgs,
    },
    #[serde(rename = "announce_peer")]
    AnnouncePeer {
        #[serde(rename = "a")]
        args: AnnouncePeerArgs,
    },
}

impl From<QueryMessageSerial> for QueryMessage {
    fn from(q: QueryMessageSerial) -> Self {
        match q {
            QueryMessageSerial::Ping { args } => Self::from(args),
            QueryMessageSerial::FindNode { args } => Self::from(args),
            QueryMessageSerial::GetPeers { args } => Self::from(args),
            QueryMessageSerial::AnnouncePeer { args } => Self::from(args),
        }
    }
}

impl From<QueryMessage> for QueryMessageSerial {
    fn from(q: QueryMessage) -> Self {
        match q {
            QueryMessage::Ping { id } => Self::Ping {
                args: PingArgs { id },
            },
            QueryMessage::FindNode { id, target } => Self::FindNode {
                args: FindNodeArgs { id, target },
            },
            QueryMessage::GetPeers { id, info_hash } => Self::GetPeers {
                args: GetPeersArgs { id, info_hash },
            },
            QueryMessage::AnnouncePeer {
                id,
                token,
                info_hash,
                port,
            } => Self::AnnouncePeer {
                args: AnnouncePeerArgs {
                    id,
                    token,
                    info_hash,
                    port,
                },
            },
        }
    }
}

impl From<PingArgs> for QueryMessage {
    fn from(a: PingArgs) -> Self {
        Self::Ping { id: a.id }
    }
}

impl From<FindNodeArgs> for QueryMessage {
    fn from(a: FindNodeArgs) -> Self {
        Self::FindNode {
            id: a.id,
            target: a.target,
        }
    }
}

impl From<GetPeersArgs> for QueryMessage {
    fn from(a: GetPeersArgs) -> Self {
        Self::GetPeers {
            id: a.id,
            info_hash: a.info_hash,
        }
    }
}

impl From<AnnouncePeerArgs> for QueryMessage {
    fn from(a: AnnouncePeerArgs) -> Self {
        Self::AnnouncePeer {
            id: a.id,
            token: a.token,
            info_hash: a.info_hash,
            port: a.port,
        }
    }
}

#[derive(Serialize, Deserialize)]
pub struct PingArgs {
    id: Id,
}

#[derive(Serialize, Deserialize)]
pub struct FindNodeArgs {
    id: Id,
    target: Id,
}

#[derive(Serialize, Deserialize)]
pub struct GetPeersArgs {
    id: Id,
    info_hash: InfoHash,
}

#[derive(Serialize, Deserialize)]
pub struct AnnouncePeerArgs {
    id: Id,
    token: Token,
    info_hash: InfoHash,
    #[serde(flatten)]
    port: Port,
}

#[derive(Copy, Clone, Debug, Serialize, Deserialize, PartialEq, Eq)]
#[serde(into = "PortSerial", try_from = "PortSerial")]
pub enum Port {
    Implicit,
    Explicit(u16),
}

impl Port {
    pub fn unwrap_or(&self, port: u16) -> u16 {
        match self {
            Port::Implicit => port,
            Port::Explicit(port) => *port,
        }
    }
}

#[derive(Serialize, Deserialize)]
struct PortSerial {
    #[serde(
        default,
        skip_serializing_if = "is_false",
        with = "serde_bool_as_0_or_not"
    )]
    implied_port: bool,
    port: u16,
}

impl TryFrom<PortSerial> for Port {
    type Error = Error;
    fn try_from(port: PortSerial) -> Result<Self> {
        if port.implied_port {
            Ok(Self::Implicit)
        } else if port.port != 0 {
            Ok(Self::Explicit(port.port))
        } else {
            Err(Error::ParseFailure("bad port".into()))
        }
    }
}

impl From<Port> for PortSerial {
    fn from(port: Port) -> Self {
        match port {
            Port::Implicit => PortSerial {
                implied_port: true,
                port: 0,
            },
            Port::Explicit(port) => PortSerial {
                implied_port: false,
                port,
            },
        }
    }
}

/// A response message
#[derive(Clone, Debug, Serialize, Deserialize, PartialEq, Eq)]
pub struct ResponseMessage {
    id: Id,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    token: Option<Token>,
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        with = "serde_compact_peers"
    )]
    values: Vec<SocketAddrV4>,
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        with = "serde_compact_node_info"
    )]
    nodes: Vec<NodeInfo>,
}

impl ResponseMessage {
    pub fn parse(&self, expected: ExpectedResponse) -> Result<Response> {
        Ok(match expected {
            ExpectedResponse::Pong => {
                if self.token.is_some() || !self.values.is_empty() || !self.nodes.is_empty() {
                    return Err(Error::ParseFailure("invalid ping response".into()));
                }
                Response::Pong { id: self.id }
            }
            ExpectedResponse::ClosestNodes => {
                if self.token.is_some() || !self.values.is_empty() || self.nodes.is_empty() {
                    return Err(Error::ParseFailure("invalid find_node response".into()));
                }
                Response::ClosestNodes {
                    id: self.id,
                    nodes: self.nodes.clone(),
                }
            }
            ExpectedResponse::GotPeers => {
                if self.token.is_none() || self.values.is_empty() == self.nodes.is_empty() {
                    return Err(Error::ParseFailure("invalid get_peers response".into()));
                }
                if self.nodes.is_empty() {
                    Response::KnownPeers {
                        id: self.id,
                        token: self.token.clone().unwrap(),
                        peers: self.values.clone(),
                    }
                } else {
                    Response::NoKnownPeers {
                        id: self.id,
                        token: self.token.clone().unwrap(),
                        nodes: self.nodes.clone(),
                    }
                }
            }
            ExpectedResponse::Announced => {
                if self.token.is_some() || !self.values.is_empty() || !self.nodes.is_empty() {
                    return Err(Error::ParseFailure("invalid announce_peer response".into()));
                }
                Response::Announced { id: self.id }
            }
        })
    }

    fn pong(id: Id) -> Self {
        Self {
            id,
            ..Self::blank()
        }
    }

    fn closest_nodes(id: Id, nodes: Vec<NodeInfo>) -> Self {
        Self {
            id,
            nodes,
            ..Self::blank()
        }
    }

    fn known_peers(id: Id, token: Token, peers: Vec<SocketAddrV4>) -> Self {
        Self {
            id,
            token: Some(token),
            values: peers,
            ..Self::blank()
        }
    }

    fn no_known_peers(id: Id, token: Token, nodes: Vec<NodeInfo>) -> Self {
        Self {
            id,
            token: Some(token),
            nodes,
            ..Self::blank()
        }
    }

    fn announced(id: Id) -> Self {
        Self {
            id,
            ..Self::blank()
        }
    }

    /// Gets the ID of the node who sent this message
    pub fn id(&self) -> Id {
        self.id
    }

    /// Gets the token sent with this message, if any
    pub fn token(&self) -> Option<&Token> {
        self.token.as_ref()
    }

    /// Gets the nodes sent with this message
    pub fn nodes(&self) -> &[NodeInfo] {
        &self.nodes
    }

    /// Gets the peers sent with this message
    pub fn peers(&self) -> &[SocketAddrV4] {
        &self.values
    }

    fn blank() -> Self {
        Self {
            id: Id::ZERO,
            token: None,
            values: vec![],
            nodes: vec![],
        }
    }

    fn set_id(&mut self, my_id: Id) {
        self.id = my_id;
    }
}

#[derive(Copy, Clone, Debug, PartialEq, Eq)]
pub enum ExpectedResponse {
    Pong,
    ClosestNodes,
    GotPeers,
    Announced,
}

#[derive(Clone, Debug, Serialize, PartialEq, Eq)]
#[serde(into = "ResponseMessage")]
pub enum Response {
    Pong {
        id: Id,
    },
    ClosestNodes {
        id: Id,
        nodes: Vec<NodeInfo>,
    },
    KnownPeers {
        id: Id,
        token: Token,
        peers: Vec<SocketAddrV4>,
    },
    NoKnownPeers {
        id: Id,
        token: Token,
        nodes: Vec<NodeInfo>,
    },
    Announced {
        id: Id,
    },
}

impl Response {
    pub fn id(&self) -> Id {
        match self {
            Response::Pong { id } => *id,
            Response::ClosestNodes { id, .. } => *id,
            Response::KnownPeers { id, .. } => *id,
            Response::NoKnownPeers { id, .. } => *id,
            Response::Announced { id } => *id,
        }
    }

    pub fn token(&self) -> Option<&Token> {
        match self {
            Response::Pong { .. } => None,
            Response::ClosestNodes { .. } => None,
            Response::KnownPeers { token, .. } => Some(token),
            Response::NoKnownPeers { token, .. } => Some(token),
            Response::Announced { .. } => None,
        }
    }
}

impl From<Response> for ResponseMessage {
    fn from(r: Response) -> ResponseMessage {
        match r {
            Response::Pong { id } => ResponseMessage::pong(id),
            Response::ClosestNodes { id, nodes } => ResponseMessage::closest_nodes(id, nodes),
            Response::KnownPeers { id, token, peers } => {
                ResponseMessage::known_peers(id, token, peers)
            }
            Response::NoKnownPeers { id, token, nodes } => {
                ResponseMessage::no_known_peers(id, token, nodes)
            }
            Response::Announced { id } => ResponseMessage::announced(id),
        }
    }
}

/// An error message
#[derive(Clone, Debug, Serialize, Deserialize, PartialEq, Eq)]
#[serde(into = "ErrorMessageSerial", try_from = "ErrorMessageSerial")]
pub struct ErrorMessage {
    code: ErrorCode,
    message: String,
}

impl ErrorMessage {
    /// Creates a new error message with a given code and message
    pub fn new<S>(code: ErrorCode, message: S) -> Self
    where
        String: From<S>,
    {
        Self {
            code,
            message: message.into(),
        }
    }

    pub fn generic<S>(message: S) -> Self
    where
        String: From<S>,
    {
        Self::new(ErrorCode::GenericError, message)
    }

    pub fn server<S>(message: S) -> Self
    where
        String: From<S>,
    {
        Self::new(ErrorCode::ServerError, message)
    }

    pub fn protocol<S>(message: S) -> Self
    where
        String: From<S>,
    {
        Self::new(ErrorCode::ProtocolError, message)
    }

    pub fn method_unknown<S>(message: S) -> Self
    where
        String: From<S>,
    {
        Self::new(ErrorCode::MethodUnknown, message)
    }

    /// Gets this message's error code
    pub fn code(&self) -> ErrorCode {
        self.code
    }

    /// Gets the error message
    pub fn message(&self) -> &str {
        &self.message
    }
}

impl StdError for ErrorMessage {}

#[derive(Serialize, Deserialize)]
#[serde(transparent)]
struct ErrorMessageSerial(Bencoded);

impl TryFrom<ErrorMessageSerial> for ErrorMessage {
    type Error = Error;
    fn try_from(e: ErrorMessageSerial) -> Result<Self> {
        match e.0 {
            Bencoded::List(l) if l.len() == 2 => Ok(Self {
                code: match &l[0] {
                    Bencoded::Int(code) => ErrorCode::from(*code),
                    _ => return Err(Error::ParseFailure("invalid error code".into())),
                },
                message: match &l[1] {
                    Bencoded::Bytes(b) => String::from_utf8_lossy(b).into(),
                    _ => return Err(Error::ParseFailure("invalid error message".into())),
                },
            }),
            _ => Err(Error::ParseFailure("invalid error payload".into())),
        }
    }
}

impl From<ErrorMessage> for ErrorMessageSerial {
    fn from(e: ErrorMessage) -> Self {
        Self(Bencoded::List(vec![
            Bencoded::Int(e.code.into()),
            Bencoded::Bytes(e.message.into()),
        ]))
    }
}

impl Display for ErrorMessage {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        write!(f, "error {}: {}", self.code, self.message)
    }
}

#[derive(Copy, Clone, Debug, Serialize, Deserialize, PartialEq, Eq)]
#[serde(into = "i64", from = "i64")]
pub enum ErrorCode {
    GenericError,
    ServerError,
    ProtocolError,
    MethodUnknown,
    Other(i64),
}

impl From<i64> for ErrorCode {
    fn from(code: i64) -> Self {
        match code {
            201 => Self::GenericError,
            202 => Self::ServerError,
            203 => Self::ProtocolError,
            204 => Self::MethodUnknown,
            code => Self::Other(code),
        }
    }
}

impl From<ErrorCode> for i64 {
    fn from(code: ErrorCode) -> Self {
        match code {
            ErrorCode::GenericError => 201,
            ErrorCode::ServerError => 202,
            ErrorCode::ProtocolError => 203,
            ErrorCode::MethodUnknown => 204,
            ErrorCode::Other(code) => code,
        }
    }
}

impl Display for ErrorCode {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        <Self as Debug>::fmt(self, f)
    }
}

mod serde_compact_node_info {
    use crate::{id::Id, node_info::NodeInfo};
    use serde::{de, Deserialize, Deserializer, Serializer};
    use serde_bytes::ByteBuf;
    use std::net::{Ipv4Addr, SocketAddrV4};

    pub fn serialize<S>(value: &[NodeInfo], ser: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        let mut bytes = Vec::with_capacity(value.len() * IPV4_COMPACT_LEN);

        for node in value.iter() {
            bytes.extend(node.id().as_ref());
            bytes.extend(&node.address().ip().octets());
            bytes.extend(&node.address().port().to_be_bytes());
        }

        ser.serialize_bytes(&bytes)
    }

    pub fn deserialize<'de, D>(de: D) -> Result<Vec<NodeInfo>, D::Error>
    where
        D: Deserializer<'de>,
    {
        let bytes = ByteBuf::deserialize(de)?;
        if bytes.len() % IPV4_COMPACT_LEN != 0 {
            return Err(de::Error::invalid_length(bytes.len(), &"multiple of 26"));
        }
        let addresses: Vec<_> = bytes
            .chunks(IPV4_COMPACT_LEN)
            .map(|c| {
                let (id, c) = c.split_at(Id::LEN);
                let (ip, port) = c.split_at(IPV4_ADDRESS_LEN);
                let ip = Ipv4Addr::new(ip[0], ip[1], ip[2], ip[3]);
                let port = u16::from_be_bytes([port[0], port[1]]);
                let mut buf = [0u8; Id::LEN];
                buf.copy_from_slice(id);
                NodeInfo::new(Id::from(buf), SocketAddrV4::new(ip, port))
            })
            .collect();
        Ok(addresses)
    }
    const IPV4_ADDRESS_LEN: usize = 4;
    const IPV4_PORT_LEN: usize = 2;
    const IPV4_COMPACT_LEN: usize = Id::LEN + IPV4_ADDRESS_LEN + IPV4_PORT_LEN;
}

mod serde_compact_peers {
    use crate::serde::serde_byte_array;
    use serde::{de, ser::SerializeSeq, Deserialize, Deserializer, Serialize, Serializer};
    use serde_bytes::ByteBuf;
    use std::net::{Ipv4Addr, SocketAddrV4};

    #[derive(Serialize)]
    #[serde(transparent)]
    struct Bytes(#[serde(with = "serde_byte_array")] [u8; IPV4_COMPACT_LEN]);

    pub fn serialize<S>(value: &[SocketAddrV4], ser: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        let mut seq = ser.serialize_seq(Some(value.len()))?;
        for addr in value.iter() {
            let mut bytes = [0u8; IPV4_COMPACT_LEN];
            let (ip, port) = bytes.split_at_mut(IPV4_ADDRESS_LEN);
            ip.copy_from_slice(&addr.ip().octets());
            port.copy_from_slice(&addr.port().to_be_bytes());
            seq.serialize_element(&Bytes(bytes))?;
        }
        seq.end()
    }

    pub fn deserialize<'de, D>(de: D) -> Result<Vec<SocketAddrV4>, D::Error>
    where
        D: Deserializer<'de>,
    {
        let bufs = <Vec<ByteBuf>>::deserialize(de)?;

        let addresses: Result<Vec<_>, _> = bufs
            .into_iter()
            .map(|buf| {
                if buf.len() != IPV4_COMPACT_LEN {
                    return Err(de::Error::invalid_length(buf.len(), &"multiple of 6"));
                }
                let (ip, port) = buf.split_at(IPV4_ADDRESS_LEN);
                let ip = Ipv4Addr::new(ip[0], ip[1], ip[2], ip[3]);
                let port = u16::from_be_bytes([port[0], port[1]]);
                Ok(SocketAddrV4::new(ip, port))
            })
            .collect();
        addresses
    }
    const IPV4_ADDRESS_LEN: usize = 4;
    const IPV4_COMPACT_LEN: usize = IPV4_ADDRESS_LEN + 2;
}

#[cfg(test)]
mod test {
    use super::*;
    use hex_literal::hex;
    use serde_bencode::{from_bytes as bdecode, to_bytes as bencode};
    use std::net::Ipv4Addr;

    #[test]
    fn messages_are_bencoded_correctly() {
        let id = Id::from(hex!("4142434445464748494a4b4c4d4e4f5051525354"));
        let id2 = Id::from(hex!("6162636465666768696a6b6c6d6e6f7071727374"));
        let id3 = InfoHash::from(hex!("4161426243634464456546664767486849694a6a"));
        let txid = TxId::from(0x30313233);
        let addr1 = SocketAddrV4::new(Ipv4Addr::new(0x58, 0x58, 0x58, 0x58), 0x5858);
        let addr2 = SocketAddrV4::new(Ipv4Addr::new(0x59, 0x59, 0x59, 0x59), 0x5959);
        let node1 = NodeInfo::new(id, addr1);
        let node2 = NodeInfo::new(id2, addr2);
        let token = Token::from(0x3736353433323130);

        let message = Message::Query {
            txid,
            message: QueryMessage::Ping { id },
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:ad2:id20:ABCDEFGHIJKLMNOPQRSTe1:q4:ping1:t4:01231:y1:qe"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);
        let message = Message::Response {
            txid,
            message: ResponseMessage::pong(id),
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:rd2:id20:ABCDEFGHIJKLMNOPQRSTe1:t4:01231:y1:re"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);

        let message = Message::Query {
            txid,
            message: QueryMessage::FindNode { id, target: id2 },
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:ad2:id20:ABCDEFGHIJKLMNOPQRST6:target20:abcdefghijklmnopqrste1:q9:find_node1:t4:01231:y1:qe"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);
        let message = Message::Response {
            txid,
            message: ResponseMessage::closest_nodes(id, vec![node1, node2]),
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:rd2:id20:ABCDEFGHIJKLMNOPQRST5:nodes52:ABCDEFGHIJKLMNOPQRSTXXXXXXabcdefghijklmnopqrstYYYYYYe1:t4:01231:y1:re"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);

        let message = Message::Query {
            txid,
            message: QueryMessage::GetPeers { id, info_hash: id3 },
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:ad2:id20:ABCDEFGHIJKLMNOPQRST9:info_hash20:AaBbCcDdEeFfGgHhIiJje1:q9:get_peers1:t4:01231:y1:qe"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);
        let message = Message::Response {
            txid,
            message: ResponseMessage::no_known_peers(id, token.clone(), vec![node1, node2]),
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:rd2:id20:ABCDEFGHIJKLMNOPQRST5:nodes52:ABCDEFGHIJKLMNOPQRSTXXXXXXabcdefghijklmnopqrstYYYYYY5:token8:76543210e1:t4:01231:y1:re"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);
        let message = Message::Response {
            txid,
            message: ResponseMessage::known_peers(id, token.clone(), vec![addr1, addr2]),
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:rd2:id20:ABCDEFGHIJKLMNOPQRST5:token8:765432106:valuesl6:XXXXXX6:YYYYYYee1:t4:01231:y1:re"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);

        let message = Message::Query {
            txid,
            message: QueryMessage::AnnouncePeer {
                id,
                token: token.clone(),
                info_hash: id3,
                port: Port::Explicit(1723),
            },
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:ad2:id20:ABCDEFGHIJKLMNOPQRST9:info_hash20:AaBbCcDdEeFfGgHhIiJj4:porti1723e5:token8:76543210e1:q13:announce_peer1:t4:01231:y1:qe"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);
        let message = Message::Query {
            txid,
            message: QueryMessage::AnnouncePeer {
                id,
                token,
                info_hash: id3,
                port: Port::Implicit,
            },
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:ad2:id20:ABCDEFGHIJKLMNOPQRST12:implied_porti1e9:info_hash20:AaBbCcDdEeFfGgHhIiJj4:porti0e5:token8:76543210e1:q13:announce_peer1:t4:01231:y1:qe"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);
        let message = Message::Response {
            txid,
            message: ResponseMessage::announced(id),
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:rd2:id20:ABCDEFGHIJKLMNOPQRSTe1:t4:01231:y1:re"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);

        let message = Message::Error {
            txid,
            message: ErrorMessage::new(ErrorCode::GenericError, "wtf"),
        };
        let buf = bencode(&message).unwrap();
        assert_eq!(
            String::from_utf8(buf.clone()).unwrap(),
            "d1:eli201e3:wtfe1:t4:01231:y1:ee"
        );
        let decoded: Message = bdecode(&buf).unwrap();
        assert_eq!(decoded, message);
    }
}
