//! DHT errors

use crate::message::ErrorMessage;
use futures::channel::mpsc::{SendError, TrySendError};
use std::io;
use thiserror::Error;

/// DHT errors
#[derive(Debug, Error)]
pub enum Error {
    /// An I/O error has occurred.
    #[error(transparent)]
    Io(#[from] io::Error),

    /// Message parsing failed
    #[error("message parsing failed: {0:}")]
    ParseFailure(String),

    /// A peer sent an error reply
    #[error("a peer sent an error reply: {0:}")]
    ErrorReply(#[from] ErrorMessage),

    /// A query has timed out
    #[error("a query has timed out")]
    Timeout,

    /// Sending a message failed
    #[error("sending a message failed")]
    SendFailure,

    /// Receiving a message failed
    #[error("receiving a message failed")]
    ReceiveFailure,
}

impl From<serde_bencode::Error> for Error {
    fn from(e: serde_bencode::Error) -> Self {
        Error::ParseFailure(e.to_string())
    }
}

impl<T> From<TrySendError<T>> for Error {
    fn from(_: TrySendError<T>) -> Self {
        Error::SendFailure
    }
}

impl From<SendError> for Error {
    fn from(_: SendError) -> Self {
        Error::SendFailure
    }
}

impl From<async_std::channel::RecvError> for Error {
    fn from(_: async_std::channel::RecvError) -> Self {
        Error::ReceiveFailure
    }
}

/// Result type for DHT errors.
pub type Result<T> = std::result::Result<T, Error>;

// TODO clean up internal errors
