//! Node directory

use crate::{
    id::{Distance, Id, IdRange},
    node::QueryHandle,
    node_info::NodeInfo,
    timer::Timer,
};
use rand::{seq::SliceRandom, thread_rng, Rng};
use std::{
    collections::BTreeMap,
    fmt::{self, Debug},
    mem::swap,
    net::SocketAddrV4,
    ops::RangeBounds,
    time::Duration,
};

pub struct Directory {
    id: Id,
    buckets: BTreeMap<Id, Bucket>,
    reverse: BTreeMap<SocketAddrV4, Id>,
}

impl Directory {
    pub fn new(my_id: Id, query_handle: QueryHandle) -> Self {
        Self {
            id: my_id,
            buckets: {
                let mut map = BTreeMap::new();
                map.insert(Id::ZERO, Bucket::new(IdRange::full(), query_handle));
                map
            },
            reverse: BTreeMap::new(),
        }
    }

    pub fn id(&self) -> Id {
        self.id
    }

    pub fn is_empty(&self) -> bool {
        for bucket in self.buckets.values() {
            if !bucket.is_empty() {
                return false;
            }
        }
        true
    }

    pub fn id_of(&self, addr: SocketAddrV4) -> Option<Id> {
        self.reverse.get(&addr).copied()
    }

    pub fn add_node(&mut self, new_node: NodeInfo) {
        let my_id = self.id;
        if my_id == new_node.id() {
            return;
        }
        let address = new_node.address();
        let id = new_node.id();
        loop {
            let bucket = self.get_bucket_mut(new_node.id());
            match bucket.add_node(new_node) {
                AddResult::Full => {
                    if bucket.range().contains(&my_id) {
                        if let Some(new_bucket) = bucket.split() {
                            self.add_bucket(new_bucket);
                        }
                        continue;
                    } else if let Some(old) = bucket.pending_node.replace(new_node) {
                        bucket.ping(old);
                    }
                }
                AddResult::Added => {
                    self.reverse.insert(address, id);
                }
                AddResult::Replaced(old) => {
                    self.reverse.remove(&old.address());
                }
                _ => {
                }
            }
            break;
        }
    }

    pub fn closest_nodes(&self, id: Id) -> Vec<NodeInfo> {
        let mut vec = vec![];
        let mut it = self.buckets.range(..=id);

        let bucket = it.next_back().expect("there are no buckets").1;
        vec.extend(bucket.nodes().filter(|n| !n.is_bad()));

        if vec.len() < Self::BUCKET_SIZE {
            let mut it = it.clone();
            if let Some((_, bucket)) = it.next() {
                vec.extend(bucket.nodes().filter(|n| !n.is_bad()));
            }

            if let Some((_, bucket)) = it.next_back() {
                vec.extend(bucket.nodes().filter(|n| !n.is_bad()));
            }
        }

        vec.sort_by_key(|n: &NodeInfo| Distance::between(n.id(), id));
        vec.truncate(Self::BUCKET_SIZE);
        vec
    }

    pub fn refresh(&mut self) {
        while let Some(bucket) = self.buckets.values_mut().find(|s| s.is_stale()) {
            bucket.refresh();
        }
    }

    pub fn mark_response(&mut self, id: Id) {
        let bucket = self.get_bucket_mut(id);
        bucket.mark_response(id);
    }

    pub fn mark_query(&mut self, id: Id) {
        let bucket = self.get_bucket_mut(id);
        bucket.mark_query(id);
    }

    pub fn mark_ping(&mut self, id: Id) {
        let bucket = self.get_bucket_mut(id);
        bucket.mark_ping(id);
    }

    fn add_bucket(&mut self, new_bucket: Bucket) {
        self.buckets.insert(new_bucket.range().first(), new_bucket);
    }

    fn get_bucket(&self, id: Id) -> &Bucket {
        self.buckets
            .range(..=id)
            .next_back()
            .expect("there are no buckets")
            .1
    }

    fn get_bucket_mut(&mut self, id: Id) -> &mut Bucket {
        self.buckets
            .range_mut(..=id)
            .next_back()
            .expect("there are no buckets")
            .1
    }

    pub const BUCKET_SIZE: usize = 8;
}

impl Debug for Directory {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.debug_struct("Directory")
            .field("id", &self.id)
            .field("nodes", &self.reverse)
            .finish()
    }
}

#[derive(Clone)]
struct Bucket {
    nodes: Vec<NodeInfo>,
    range: IdRange,
    refresh_timer: Timer,
    pending_node: Option<NodeInfo>,
    query_handle: QueryHandle,
}

impl Bucket {
    pub fn new(range: IdRange, query_handle: QueryHandle) -> Self {
        Self {
            nodes: vec![],
            range,
            pending_node: None,
            refresh_timer: Timer::start(Self::REFRESH_TIME),
            query_handle,
        }
    }

    pub fn range(&self) -> IdRange {
        self.range
    }

    pub fn nodes(&self) -> impl Iterator<Item = &NodeInfo> {
        self.nodes.iter()
    }

    pub fn find_mut(&mut self, id: Id) -> Option<&mut NodeInfo> {
        self.nodes.iter_mut().find(|n| n.id() == id)
    }

    pub fn split(&mut self) -> Option<Bucket> {
        if self.range.usable_bits() <= 3 {
            return None;
        }
        if let Some(node) = self.pending_node.take() {
            self.ping(node);
        }
        let (left, right) = self.range.split();
        self.range = left;
        let mut new_bucket = Bucket::new(right, self.query_handle.clone());
        new_bucket
            .nodes
            .extend(self.nodes.drain_filter(|n| right.contains(&n.id())));
        new_bucket.touch();
        Some(new_bucket)
    }

    pub fn len(&self) -> usize {
        self.nodes.len()
    }

    pub fn is_empty(&self) -> bool {
        self.nodes.is_empty()
    }

    pub fn random_node(&self) -> Option<&NodeInfo> {
        self.nodes.choose(&mut thread_rng())
    }

    pub fn is_stale(&self) -> bool {
        self.refresh_timer.is_expired()
    }

    pub fn add_node(&mut self, mut node: NodeInfo) -> AddResult {
        let id = node.id();
        if !self.range().contains(&id) {
            return AddResult::Invalid;
        }
        if let Some(old) = self.find_mut(id) {
            old.merge(node);
            self.touch();
            return AddResult::Added;
        }
        if self.nodes.len() < Directory::BUCKET_SIZE {
            self.nodes.push(node);
            self.touch();
            return AddResult::Added;
        }
        if let Some(bad_node) = self.nodes.iter_mut().find(|n| n.is_bad()) {
            swap(bad_node, &mut node);
            self.touch();
            return AddResult::Replaced(node);
        }
        let questionable = self.nodes.iter_mut().filter(|n| n.is_questionable());
        for questionable_node in questionable {
            if questionable_node.can_ping() {
                let questionable_node = *questionable_node;
                self.ping(questionable_node);
                if let Some(old) = self.pending_node.replace(node) {
                    self.ping(old);
                }
                return AddResult::Pinging;
            }
        }
        AddResult::Full
    }

    pub fn refresh(&mut self) {
        if let Some(node) = self.random_node() {
            let node = *node;
            let target = thread_rng().gen_range(self.range());
            self.find_node(node, target);
            self.touch();
        }
    }

    pub fn mark_response(&mut self, id: Id) {
        if let Some(n) = self.nodes.iter_mut().find(|n| n.id() == id) {
            n.mark_response();
        }
    }

    pub fn mark_query(&mut self, id: Id) {
        if let Some(n) = self.nodes.iter_mut().find(|n| n.id() == id) {
            n.mark_query();
        }
    }

    pub fn mark_ping(&mut self, id: Id) {
        if let Some(n) = self.nodes.iter_mut().find(|n| n.id() == id) {
            n.mark_ping();
        }
    }

    fn ping(&mut self, node: NodeInfo) {
        let _ = self.query_handle.ping(node.address());
    }

    fn find_node(&mut self, node: NodeInfo, target: Id) {
        let _ = self.query_handle.find_node(target, node.address());
    }

    fn touch(&mut self) {
        self.refresh_timer.restart();
    }

    const REFRESH_TIME: Duration = Duration::from_secs(15 * 60);
}

impl Debug for Bucket {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.debug_struct("Bucket")
            .field("nodes", &self.nodes)
            .field("range", &self.range)
            .field("pending_node", &self.pending_node)
            .finish()
    }
}

enum AddResult {
    Added,
    Replaced(NodeInfo),
    Pinging,
    Full,
    Invalid,
}
