//! Transport abstraction

use crate::Result;
use std::{
    future::Future,
    net::SocketAddrV4,
    pin::Pin,
    task::{Context, Poll},
};

/// DHT transport abstraction
pub trait Transport {
    /// Attempt to send a datagram
    fn poll_send(
        self: Pin<&mut Self>,
        cx: &mut Context,
        buf: &[u8],
        addr: SocketAddrV4,
    ) -> Poll<Result<usize>>;

    /// Attempt to receive a datagram
    fn poll_receive(
        self: Pin<&mut Self>,
        cx: &mut Context,
        buf: &mut [u8],
    ) -> Poll<Result<(usize, SocketAddrV4)>>;
}

impl<T> Transport for &mut T
where
    T: Transport + Unpin + ?Sized,
{
    fn poll_send(
        mut self: Pin<&mut Self>,
        cx: &mut Context,
        buf: &[u8],
        addr: SocketAddrV4,
    ) -> Poll<Result<usize>> {
        Pin::new(&mut **self).poll_send(cx, buf, addr)
    }
    fn poll_receive(
        mut self: Pin<&mut Self>,
        cx: &mut Context,
        buf: &mut [u8],
    ) -> Poll<Result<(usize, SocketAddrV4)>> {
        Pin::new(&mut **self).poll_receive(cx, buf)
    }
}

pub trait TransportExt: Transport {
    fn send<'a>(&'a mut self, buf: &'a [u8], addr: SocketAddrV4) -> Send<'a, Self>
    where
        Self: Unpin,
    {
        Send::new(self, buf, addr)
    }
    fn receive<'a>(&'a mut self, buf: &'a mut [u8]) -> Receive<'a, Self>
    where
        Self: Unpin,
    {
        Receive::new(self, buf)
    }
}

impl<T> TransportExt for T where T: Transport + ?Sized {}

pub struct Send<'a, T>
where
    T: ?Sized,
{
    writer: &'a mut T,
    buf: &'a [u8],
    addr: SocketAddrV4,
}

impl<T> Unpin for Send<'_, T> where T: Unpin + ?Sized {}

impl<'a, T> Send<'a, T>
where
    T: Transport + Unpin + ?Sized,
{
    fn new(writer: &'a mut T, buf: &'a [u8], addr: SocketAddrV4) -> Self {
        Self { writer, buf, addr }
    }
}

impl<'a, T> Future for Send<'a, T>
where
    T: Transport + Unpin + ?Sized,
{
    type Output = Result<usize>;

    fn poll(mut self: Pin<&mut Self>, cx: &mut Context) -> Poll<Self::Output> {
        let this = &mut *self;
        Pin::new(&mut this.writer).poll_send(cx, this.buf, this.addr)
    }
}

pub struct Receive<'a, T>
where
    T: ?Sized,
{
    reader: &'a mut T,
    buf: &'a mut [u8],
}

impl<T> Unpin for Receive<'_, T> where T: ?Sized + Unpin {}

impl<'a, T> Receive<'a, T>
where
    T: Transport + ?Sized + Unpin,
{
    fn new(reader: &'a mut T, buf: &'a mut [u8]) -> Self {
        Self { reader, buf }
    }
}

impl<'a, T> Future for Receive<'a, T>
where
    T: Transport + ?Sized + Unpin,
{
    type Output = Result<(usize, SocketAddrV4)>;

    fn poll(mut self: Pin<&mut Self>, cx: &mut Context) -> Poll<Self::Output> {
        let this = &mut *self;
        Pin::new(&mut this.reader).poll_receive(cx, this.buf)
    }
}
