//! Testing network

use crate::{message::Message, Error, Result, Transport};
use futures::{
    channel::mpsc::{unbounded, UnboundedReceiver, UnboundedSender},
    ready,
    sink::SinkExt,
    stream::{Stream, StreamExt},
};
use pin_project::pin_project;
use rand::random;
use serde_bencode::from_bytes as bdecode;
use std::{
    cmp::min as at_most,
    collections::BTreeMap,
    net::{Ipv4Addr, SocketAddrV4},
    pin::Pin,
    task::{
        Context,
        Poll::{self, Ready},
    },
};

pub struct TestNet {
    receiver: UnboundedReceiver<Packet>,
    peer_sender: UnboundedSender<Packet>,
    senders: BTreeMap<SocketAddrV4, UnboundedSender<Packet>>,
}

impl TestNet {
    pub fn new() -> Self {
        let (peer_sender, receiver) = unbounded();
        Self {
            receiver,
            peer_sender,
            senders: BTreeMap::new(),
        }
    }

    pub fn join(&mut self) -> TestSocket {
        let addr = SocketAddrV4::new(
            Ipv4Addr::new(random(), random(), random(), random()),
            random(),
        );
        let (peer, sender) = TestSocket::new(addr, self.peer_sender.clone());
        self.senders.insert(addr, sender);
        peer
    }

    pub async fn run(mut self) {
        loop {
            while let Some(packet) = self.receiver.next().await {
                if let Some(sender) = self.senders.get_mut(&packet.destination) {
                    let _ = sender.send(packet).await;
                }
            }
        }
    }
}

#[pin_project]
pub struct TestSocket {
    addr: SocketAddrV4,
    #[pin]
    sender: UnboundedSender<Packet>,
    #[pin]
    receiver: UnboundedReceiver<Packet>,
}

impl TestSocket {
    fn new(addr: SocketAddrV4, sender: UnboundedSender<Packet>) -> (Self, UnboundedSender<Packet>) {
        let (my_sender, receiver) = unbounded();
        (
            Self {
                addr,
                sender,
                receiver,
            },
            my_sender,
        )
    }

    pub fn address(&self) -> SocketAddrV4 {
        self.addr
    }
}

impl Transport for TestSocket {
    fn poll_send(
        mut self: Pin<&mut Self>,
        cx: &mut Context,
        buf: &[u8],
        addr: SocketAddrV4,
    ) -> Poll<Result<usize>> {
        let mut this = self.as_mut().project();
        ready!(this.sender.poll_ready(cx))?;
        let packet = Packet {
            source: *this.addr,
            destination: addr,
            payload: buf.to_vec(),
        };
        println!(" # # # PACKET {}", packet.to_string());
        this.sender.start_send(packet)?;
        Ready(Ok(buf.len()))
    }

    fn poll_receive(
        mut self: Pin<&mut Self>,
        cx: &mut Context,
        buf: &mut [u8],
    ) -> Poll<Result<(usize, SocketAddrV4)>> {
        let this = self.as_mut().project();
        Ready(match ready!(this.receiver.poll_next(cx)) {
            Some(packet) => {
                let n = packet.payload.len();
                let to_read = at_most(buf.len(), n);
                buf[..to_read].copy_from_slice(&packet.payload[..to_read]);
                Ok((to_read, packet.source))
            }
            None => Err(Error::ReceiveFailure),
        })
    }
}

struct Packet {
    source: SocketAddrV4,
    destination: SocketAddrV4,
    payload: Vec<u8>,
}

impl Packet {
    fn to_string(&self) -> String {
        let payload = bdecode::<Message>(&self.payload).unwrap();
        format!(
            "[ from {:>21} to {:>21}: {:?} ]",
            self.source, self.destination, payload
        )
    }
}
