//! The actual hash table

use crate::{id::Id, timer::Timer};
use std::{
    cmp::Ordering,
    collections::{BTreeMap, BTreeSet},
    net::SocketAddrV4,
    time::Duration,
};

pub struct Table {
    map: BTreeMap<Id, BTreeSet<PeerInfo>>,
}

impl Table {
    pub fn new() -> Self {
        Self {
            map: BTreeMap::new(),
        }
    }

    pub fn insert(&mut self, id: Id, address: SocketAddrV4) {
        let info = PeerInfo {
            address,
            expire_timer: Timer::new(Self::EXPIRE_TIMER),
        };
        self.map
            .entry(id)
            .or_insert_with(BTreeSet::new)
            .insert(info);
    }

    pub fn get(&self, id: &Id) -> Option<Vec<PeerInfo>> {
        self.map
            .get(id)
            .map(|s| s.iter().cloned().collect::<Vec<_>>())
    }

    pub fn get_addresses(&self, id: &Id) -> Option<Vec<SocketAddrV4>> {
        self.get(id)
            .map(|v| v.iter().map(PeerInfo::address).collect())
    }

    pub fn expire(&mut self) {
        self.map.drain_filter(|_, infos| {
            infos.drain_filter(|p| p.is_expired());
            infos.is_empty()
        });
    }

    const EXPIRE_TIMER: Duration = Duration::from_secs(30 * 60);
}

#[derive(Copy, Clone, Debug)]
pub struct PeerInfo {
    address: SocketAddrV4,
    expire_timer: Timer,
}

impl PartialOrd for PeerInfo {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.address.partial_cmp(&other.address)
    }
}

impl Ord for PeerInfo {
    fn cmp(&self, other: &Self) -> Ordering {
        self.address.cmp(&other.address)
    }
}

impl PartialEq for PeerInfo {
    fn eq(&self, other: &Self) -> bool {
        self.address.eq(&other.address)
    }
}

impl Eq for PeerInfo {}

impl PeerInfo {
    pub fn address(&self) -> SocketAddrV4 {
        self.address
    }

    pub fn is_expired(&self) -> bool {
        self.expire_timer.is_expired()
    }
}
