//! Serialization and deserialization utilities

pub mod serde_byte_array {
    use serde::{
        de::{self, Visitor},
        Deserializer, Serializer,
    };
    use std::fmt;

    pub fn serialize<S, const N: usize>(value: &[u8; N], ser: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        ser.serialize_bytes(value.as_ref())
    }

    pub fn deserialize<'de, D, const N: usize>(de: D) -> Result<[u8; N], D::Error>
    where
        D: Deserializer<'de>,
    {
        de.deserialize_bytes(ByteArrayVisitor)
    }

    struct ByteArrayVisitor<const N: usize>;

    impl<'de, const N: usize> Visitor<'de> for ByteArrayVisitor<N> {
        type Value = [u8; N];

        fn expecting(&self, f: &mut fmt::Formatter) -> fmt::Result {
            write!(f, "byte array of length {}", N)
        }

        fn visit_bytes<E>(self, v: &[u8]) -> Result<Self::Value, E>
        where
            E: de::Error,
        {
            if v.len() != N {
                return Err(de::Error::invalid_length(v.len(), &""));
            }
            let mut buf = [0u8; N];
            buf.copy_from_slice(v);
            Ok(buf)
        }

        fn visit_borrowed_bytes<E>(self, v: &'de [u8]) -> Result<Self::Value, E>
        where
            E: de::Error,
        {
            self.visit_bytes(v)
        }

        fn visit_byte_buf<E>(self, v: Vec<u8>) -> Result<Self::Value, E>
        where
            E: de::Error,
        {
            self.visit_bytes(&v)
        }

        fn visit_str<E>(self, s: &str) -> Result<Self::Value, E>
        where
            E: de::Error,
        {
            self.visit_bytes(s.as_bytes())
        }

        fn visit_borrowed_str<E>(self, s: &'de str) -> Result<Self::Value, E>
        where
            E: de::Error,
        {
            self.visit_str(s)
        }

        fn visit_string<E>(self, s: String) -> Result<Self::Value, E>
        where
            E: de::Error,
        {
            self.visit_str(&s)
        }
    }
}

pub mod serde_bool_as_0_or_not {
    use serde::{Deserialize, Deserializer, Serializer};

    pub fn serialize<S>(value: &bool, ser: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        ser.serialize_u8(if *value { 1 } else { 0 })
    }

    pub fn deserialize<'de, D>(de: D) -> Result<bool, D::Error>
    where
        D: Deserializer<'de>,
    {
        Ok(u8::deserialize(de)? != 0)
    }
}

pub fn is_false(b: &bool) -> bool {
    !*b
}
