//! Node information

use crate::{id::Id, timer::Timer};
use std::{
    borrow::Borrow,
    cmp::Ordering,
    net::SocketAddrV4,
    time::{Duration, Instant},
};

#[derive(Copy, Clone, Debug)]
pub struct NodeInfo {
    id: Id,
    address: SocketAddrV4,
    ever_responded: bool,
    response_timer: Timer,
    query_timer: Timer,
    ping_timer: Timer,
    failure_streak: u64,
}

impl NodeInfo {
    pub fn new(id: Id, address: SocketAddrV4) -> Self {
        Self {
            id,
            address,
            ever_responded: false,
            response_timer: Timer::new(Self::INACTIVITY_THRESHOLD),
            query_timer: Timer::new(Self::INACTIVITY_THRESHOLD),
            ping_timer: Timer::new(Self::PING_COOLDOWN),
            failure_streak: 0,
        }
    }

    pub fn with_id(base: Self, id: Id) -> Self {
        NodeInfo { id, ..base }
    }

    pub fn id(&self) -> Id {
        self.id
    }

    pub fn address(&self) -> SocketAddrV4 {
        self.address
    }

    pub fn is_bad(&self) -> bool {
        matches!(self.status(), NodeStatus::Bad)
    }

    pub fn is_questionable(&self) -> bool {
        matches!(self.status(), NodeStatus::Questionable)
    }

    pub fn is_good(&self) -> bool {
        matches!(self.status(), NodeStatus::Good)
    }

    pub fn can_ping(&self) -> bool {
        self.ping_timer.is_expired()
    }

    fn status(&self) -> NodeStatus {
        let now = Instant::now();
        if self.response_timer.is_expired() || self.ever_responded && self.query_timer.is_expired()
        {
            NodeStatus::Good
        } else if self.failure_streak > Self::MAX_QUERY_FAILURES {
            NodeStatus::Bad
        } else {
            NodeStatus::Questionable
        }
    }

    pub fn mark_response(&mut self) {
        self.ever_responded = true;
        self.response_timer.restart();
        self.failure_streak = 0;
    }

    pub fn mark_query(&mut self) {
        self.query_timer.restart();
        self.ping_timer.restart();
        self.failure_streak += 1;
    }

    pub fn merge(&mut self, other: Self) {
        assert!(self.id == other.id);
        self.ever_responded |= other.ever_responded;
        self.response_timer.postpone(&other.response_timer);
        self.query_timer.postpone(&other.query_timer);
        self.failure_streak = other.failure_streak;
        self.address = other.address;
    }

    const PING_COOLDOWN: Duration = Duration::from_secs(15);
    const INACTIVITY_THRESHOLD: Duration = Duration::from_secs(15 * 60);
    const MAX_QUERY_FAILURES: u64 = 3;
}

impl PartialOrd for NodeInfo {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.id.partial_cmp(&other.id)
    }
}

impl Ord for NodeInfo {
    fn cmp(&self, other: &Self) -> Ordering {
        self.id.cmp(&other.id)
    }
}

impl PartialEq for NodeInfo {
    fn eq(&self, other: &Self) -> bool {
        self.id.eq(&other.id)
    }
}

impl Eq for NodeInfo {}

impl Borrow<Id> for NodeInfo {
    fn borrow(&self) -> &Id {
        &self.id
    }
}

#[derive(Copy, Clone)]
enum NodeStatus {
    Good,
    Questionable,
    Bad,
}
