//! Kademlia errors

use crate::message::ErrorMessage;
use futures::{
    channel::mpsc::{SendError, TrySendError},
    task::SpawnError,
};
use std::io;
use thiserror::Error;

/// Kademlia errors
#[derive(Debug, Error)]
pub enum Error {
    /// An I/O error has occurred.
    #[error(transparent)]
    Io(#[from] io::Error),

    /// Message parsing failed
    #[error("message parsing failed: {0:}")]
    ParseFailure(String),

    /// A peer sent an error reply
    #[error("a peer sent an error reply: {0:}")]
    ErrorReply(#[from] ErrorMessage),

    /// Spawning a task failed
    #[error("spawning a task failed: {0:}")]
    SpawnFailure(#[from] SpawnError),

    /// A query has timed out
    #[error("a query has timed out")]
    Timeout,

    /// An invalid transaction ID was provided
    #[error("an invalid transaction ID was provided")]
    InvalidTransaction,

    /// Sending a message failed
    #[error("sending a message failed")]
    SendFailure,

    /// Bootstrapping failed
    #[error("bootstrapping failed")]
    BootstrapFailure,

    #[cfg(test)]
    #[error("receiving from a channel failed")]
    ReceiveFailure,
}

impl From<serde_bencode::Error> for Error {
    fn from(e: serde_bencode::Error) -> Self {
        Error::ParseFailure(e.to_string())
    }
}

impl<T> From<TrySendError<T>> for Error {
    fn from(e: TrySendError<T>) -> Self {
        Error::SendFailure
    }
}

impl From<SendError> for Error {
    fn from(e: SendError) -> Self {
        Error::SendFailure
    }
}

/// Result type for Kademlia errors.
pub type Result<T> = std::result::Result<T, Error>;
