//! Node directory

use crate::{
    id::{Distance, Id, IdRange},
    node::QueryHandle,
    node_info::NodeInfo,
    timer::Timer,
};
use rand::{seq::SliceRandom, thread_rng, Rng};
use std::{
    collections::BTreeMap,
    fmt::{self, Debug},
    ops::RangeBounds,
    time::Duration,
};

#[derive(Debug)]
pub struct Directory {
    id: Id,
    buckets: BTreeMap<Id, Bucket>,
}

impl Directory {
    pub fn new(my_id: Id, query_handle: QueryHandle) -> Self {
        Self {
            id: my_id,
            buckets: {
                let mut map = BTreeMap::new();
                map.insert(Id::ZERO, Bucket::new(IdRange::full(), query_handle));
                map
            },
        }
    }

    pub fn id(&self) -> Id {
        self.id
    }

    pub fn add_node(&mut self, new_node: NodeInfo) {
        let my_id = self.id;
        if my_id == new_node.id() {
            return;
        }
        loop {
            let bucket = self.get_bucket_mut(new_node.id());
            if let AddResult::Full = bucket.add_node(new_node) {
                if bucket.range().contains(&my_id) {
                    if let Some(new_bucket) = bucket.split() {
                        self.add_bucket(new_bucket);
                        // try again after splitting
                        continue;
                    }
                }
            }
            break;
        }
    }

    pub fn closest_nodes(&self, id: Id) -> Vec<NodeInfo> {
        let mut vec = vec![];
        let mut it = self.buckets.range(..=id);

        let bucket = it.next_back().expect("there are no buckets").1;
        vec.extend(bucket.nodes().filter(|n| !n.is_bad()));

        if vec.len() < Self::BUCKET_SIZE {
            let mut it = it.clone();
            if let Some((_, bucket)) = it.next() {
                vec.extend(bucket.nodes().filter(|n| !n.is_bad()));
            }

            if let Some((_, bucket)) = it.next_back() {
                vec.extend(bucket.nodes().filter(|n| !n.is_bad()));
            }
        }

        vec.sort_by_key(|n: &NodeInfo| Distance::between(n.id(), id));
        vec.truncate(Self::BUCKET_SIZE);
        vec
    }

    pub fn refresh(&mut self) {
        while let Some(bucket) = self.buckets.values_mut().find(|s| s.is_stale()) {
            bucket.refresh();
        }
    }

    fn add_bucket(&mut self, new_bucket: Bucket) {
        self.buckets.insert(new_bucket.range().first(), new_bucket);
    }

    fn get_bucket(&self, id: Id) -> &Bucket {
        self.buckets
            .range(..=id)
            .next_back()
            .expect("there are no buckets")
            .1
    }

    fn get_bucket_mut(&mut self, id: Id) -> &mut Bucket {
        self.buckets
            .range_mut(..=id)
            .next_back()
            .expect("there are no buckets")
            .1
    }

    pub const BUCKET_SIZE: usize = 8;
}

#[derive(Clone)]
struct Bucket {
    nodes: Vec<NodeInfo>,
    range: IdRange,
    refresh_timer: Timer,
    pending_node: Option<NodeInfo>,
    query_handle: QueryHandle,
}

impl Bucket {
    pub fn new(range: IdRange, query_handle: QueryHandle) -> Self {
        Self {
            nodes: vec![],
            range,
            pending_node: None,
            refresh_timer: Timer::new(Self::REFRESH_TIME),
            query_handle,
        }
    }

    pub fn range(&self) -> IdRange {
        self.range
    }

    pub fn nodes(&self) -> impl Iterator<Item = &NodeInfo> {
        self.nodes.iter()
    }

    pub fn find_mut(&mut self, id: Id) -> Option<&mut NodeInfo> {
        self.nodes.iter_mut().find(|n| n.id() == id)
    }

    pub fn split(&mut self) -> Option<Bucket> {
        if self.range.usable_bits() <= 3 {
            return None;
        }
        if let Some(node) = self.pending_node.take() {
            self.ping(node);
        }
        let (left, right) = self.range.split();
        self.range = left;
        let mut new_bucket = Bucket::new(right, self.query_handle.clone());
        new_bucket
            .nodes
            .extend(self.nodes.drain_filter(|n| right.contains(&n.id())));
        new_bucket.touch();
        Some(new_bucket)
    }

    pub fn len(&self) -> usize {
        self.nodes.len()
    }

    pub fn is_empty(&self) -> bool {
        self.nodes.is_empty()
    }

    pub fn random_node(&self) -> &NodeInfo {
        self.nodes
            .choose(&mut thread_rng())
            .expect("bucket was empty")
    }

    pub fn is_stale(&self) -> bool {
        self.refresh_timer.is_expired()
    }

    pub fn add_node(&mut self, new_node: NodeInfo) -> AddResult {
        let id = new_node.id();
        if !self.range().contains(&id) {
            return AddResult::Invalid;
        }
        if let Some(old) = self.find_mut(id) {
            old.merge(new_node);
            self.touch();
            return AddResult::Added;
        }
        if self.nodes.len() < Directory::BUCKET_SIZE {
            self.nodes.push(new_node);
            self.touch();
            return AddResult::Added;
        }
        if let Some(node) = self.nodes.iter_mut().find(|n| n.is_bad()) {
            *node = new_node;
            self.touch();
            return AddResult::Added;
        }
        let questionable = self.nodes.iter_mut().filter(|n| n.is_questionable());
        for node in questionable {
            if node.can_ping() {
                let node = *node;
                self.ping(node);
                if let Some(old) = self.pending_node.replace(new_node) {
                    self.ping(old);
                }
                return AddResult::Pinging;
            }
        }
        AddResult::Full
    }

    pub fn refresh(&mut self) {
        let target = thread_rng().gen_range(self.range());
        let node = *self.random_node();
        self.find_node(node, target);
        self.touch();
    }

    fn ping(&mut self, node: NodeInfo) {
        let _ = self.query_handle.ping(node.address());
    }

    fn find_node(&mut self, node: NodeInfo, target: Id) {
        let _ = self.query_handle.find_node(target, node.address());
    }

    fn touch(&mut self) {
        self.refresh_timer.restart();
    }

    const REFRESH_TIME: Duration = Duration::from_secs(15 * 60);
}

impl Debug for Bucket {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.debug_struct("Bucket")
            .field("nodes", &self.nodes)
            .field("range", &self.range)
            .field("refresh_timer", &self.refresh_timer)
            .field("pending_node", &self.pending_node)
            .finish()
    }
}

enum AddResult {
    Added,
    Pinging,
    Full,
    Invalid,
}
