/*
 * Copyright Stalwart Labs, Minter Ltd. See the COPYING
 * file at the top-level directory of this distribution.
 *
 * Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
 * https://www.apache.org/licenses/LICENSE-2.0> or the MIT license
 * <LICENSE-MIT or https://opensource.org/licenses/MIT>, at your
 * option. This file may not be copied, modified, or distributed
 * except according to those terms.
 */

use std::char::REPLACEMENT_CHARACTER;

pub fn add_html_token(result: &mut String, token: &[u8], add_space: bool) {
    if add_space {
        result.push(' ');
    }

    if let [b'&', entity @ .., b';'] = token {
        let mut entity_code: u32 = 0;
        if let [b'#', code @ ..] = entity {
            let (code, radix) = if let [b'x' | b'X', code @ ..] = code {
                (code, 16)
            } else {
                (code, 10)
            };

            entity_code = std::str::from_utf8(code)
                .map_or(0, |code| u32::from_str_radix(code, radix).unwrap_or(0));
        } else if (2..=31).contains(&entity.len()) {
            let mut hash = entity.len() as u32;

            /* Safe because ENTITY_HASH's size is 260 (u8::MAX + 5)
            and ENTITY_MAP's size is 18016 (18079 - 64 + 1) */
            debug_assert_eq!(ENTITY_HASH.len(), u8::MAX as usize + 5);
            debug_assert_eq!(ENTITY_MAP.len(), 18079 - 64 + 1);

            unsafe {
                for (pos, ch) in entity.iter().enumerate() {
                    match pos {
                        0 | 5 | 6 | 9 | 11 => {
                            hash += *ENTITY_HASH.get_unchecked(*ch as usize);
                        }
                        1 => {
                            hash += *ENTITY_HASH.get_unchecked(*ch as usize + 4);
                        }
                        2 | 4 => {
                            hash += *ENTITY_HASH.get_unchecked(*ch as usize + 1);
                        }
                        3 => {
                            hash += *ENTITY_HASH.get_unchecked(*ch as usize + 3);
                        }
                        _ => (),
                    }
                    if pos == entity.len() - 1 {
                        hash += *ENTITY_HASH.get_unchecked(*ch as usize);
                    }
                }

                if (64..=18079).contains(&hash) {
                    entity_code = *ENTITY_MAP.get_unchecked((hash - 64) as usize);
                }
            }
        }

        if entity_code > 0 {
            result.push(char::from_u32(entity_code).unwrap_or(REPLACEMENT_CHARACTER));
            return;
        }
    }

    // Safe because the input string is tokenized by ASCII chars, therefore the resulting
    // u8 array is UTF-8 safe.
    result.push_str(unsafe { std::str::from_utf8_unchecked(token) });
}

pub fn html_to_text(input: &str) -> String {
    let mut result = String::with_capacity(input.len());
    let input = input.as_bytes();

    let mut in_tag = false;
    let mut in_head = false;
    let mut in_comment = false;

    let mut is_token_start = true;
    let mut is_after_space = false;
    let mut is_tag_close = false;
    let mut is_new_line = true;

    let mut token_start = 0;
    let mut token_end = 0;

    let mut tag_token_pos = 0;
    let mut comment_pos = 0;

    for (pos, ch) in input.iter().enumerate() {
        if !in_comment {
            match ch {
                b'<' => {
                    if !in_tag && !in_head && !is_token_start {
                        add_html_token(
                            &mut result,
                            &input[token_start..token_end + 1],
                            is_after_space,
                        );
                        is_after_space = false;
                    }

                    tag_token_pos = 0;
                    in_tag = true;
                    is_token_start = true;
                    is_tag_close = false;
                    continue;
                }
                b'>' if in_tag => {
                    if tag_token_pos == 1 {
                        match input.get(token_start..token_end + 1) {
                            Some(tag)
                                if tag.eq_ignore_ascii_case(b"br")
                                    || (tag.eq_ignore_ascii_case(b"p") && is_tag_close) =>
                            {
                                result.push('\n');
                                is_after_space = false;
                                is_new_line = true;
                            }
                            Some(tag) if tag.eq_ignore_ascii_case(b"head") => {
                                in_head = !is_tag_close;
                            }
                            _ => (),
                        }
                    }

                    in_tag = false;
                    is_token_start = true;

                    continue;
                }
                b'/' if in_tag => {
                    if tag_token_pos == 0 {
                        is_tag_close = true;
                    }
                    continue;
                }
                b'!' if in_tag && tag_token_pos == 0 => {
                    if let Some(b"--") = input.get(pos + 1..pos + 3) {
                        in_comment = true;
                        continue;
                    }
                }
                b' ' | b'\t' | b'\r' | b'\n' => {
                    if !in_tag && !in_head {
                        if !is_token_start {
                            add_html_token(
                                &mut result,
                                &input[token_start..token_end + 1],
                                is_after_space && !is_new_line,
                            );
                            is_new_line = false;
                        }
                        is_after_space = true;
                    }

                    is_token_start = true;

                    continue;
                }
                b'&' if !in_tag && !is_token_start && !in_head => {
                    add_html_token(
                        &mut result,
                        &input[token_start..token_end + 1],
                        is_after_space && !is_new_line,
                    );
                    is_new_line = false;
                    is_token_start = true;
                    is_after_space = false;
                }
                b';' if !in_tag && !is_token_start && !in_head => {
                    add_html_token(
                        &mut result,
                        &input[token_start..pos + 1],
                        is_after_space && !is_new_line,
                    );
                    is_token_start = true;
                    is_after_space = false;
                    is_new_line = false;
                    continue;
                }
                _ => (),
            }
            if is_token_start {
                token_start = pos;
                is_token_start = false;
                if in_tag {
                    tag_token_pos += 1;
                }
            }
            token_end = pos;
        } else {
            match ch {
                b'-' => comment_pos += 1,
                b'>' if comment_pos == 2 => {
                    comment_pos = 0;
                    in_comment = false;
                    in_tag = false;
                    is_token_start = true;
                }
                _ => comment_pos = 0,
            }
        }
    }

    if !in_tag && !is_token_start && !in_head {
        add_html_token(
            &mut result,
            &input[token_start..token_end + 1],
            is_after_space && !is_new_line,
        );
    }

    result.shrink_to_fit();
    result
}

pub fn text_to_html(input: &str) -> String {
    let input = input.as_bytes();
    let mut result = Vec::with_capacity(input.len() + 26);

    result.extend_from_slice(b"<html><body>");

    for ch in input {
        match ch {
            b'\n' => result.extend_from_slice(b"<br/>"),
            b'<' => result.extend_from_slice(b"&lt;"),
            b'\r' => (),
            _ => result.push(*ch),
        }
    }
    result.extend_from_slice(b"</body></html>");

    unsafe { String::from_utf8_unchecked(result) }
}

#[cfg(test)]
mod tests {

    use crate::decoders::html::{add_html_token, html_to_text, text_to_html};

    #[test]
    fn convert_text_to_html() {
        let inputs = [
            (
                "hello\nworld\n",
                "<html><body>hello<br/>world<br/></body></html>",
            ),
            ("using <>\n", "<html><body>using &lt;><br/></body></html>"),
        ];

        for input in inputs {
            assert_eq!(text_to_html(input.0), input.1);
        }
    }

    #[test]
    fn convert_html_to_text() {
        let inputs = [
            ("<html>hello<br/>world<br/></html>", "hello\nworld\n"),
            ("<html>using &lt;><br/></html>", "using <>\n"),
            ("test <not br/>tag<br />", "test tag\n"),
            ("<>< ><tag\n/>>hello    world< br \n />", ">hello world\n"),
            (
                concat!(
                    "<head><title>ignore head</title><not head>xyz</not head></head>",
                    "<h1>&lt;body&gt;</h1>"
                ),
                "<body>",
            ),
            (
                concat!(
                    "<p>what is &heartsuit;?</p><p>&#x000DF;&Abreve;&#914;&gamma; ",
                    "don&apos;t hurt me.</p>"
                ),
                "what is ♥?\nßĂΒγ don't hurt me.\n",
            ),
            (
                concat!(
                    "<!--[if mso]><style type=\"text/css\">body, table, td, a, p, ",
                    "span, ul, li {font-family: Arial, sans-serif!important;}</style><![endif]-->",
                    "this is <!-- <> < < < < ignore  > -> here -->the actual<!--> text"
                ),
                "this is the actual text",
            ),
            (
                "   < p >  hello < / p > < p > world < / p >   !!! < br > ",
                "hello\nworld\n!!!\n",
            ),
            (
                " <p>please unsubscribe <a href=#>here</a>.</p> ",
                "please unsubscribe here.\n",
            ),
        ];

        for input in inputs {
            assert_eq!(html_to_text(input.0), input.1, "Failed for '{:?}'", input.0);
        }
    }

    #[test]
    fn convert_html_entity() {
        let inputs = [
            ("&lt;", "<"),
            ("&#32;", " "),
            ("&#x20;", " "),
            ("&nbsp;", "\u{a0}"),
            ("&CounterClockwiseContourIntegral;", "∳"),
            ("&curvearrowright;", "↷"),
            ("&rarr;", "→"),
            ("&#xFFFFFFF;", "�"),
            ("&hmmm", "&hmmm"),
        ];

        for input in inputs {
            let mut result = String::with_capacity(input.0.len());
            add_html_token(&mut result, input.0.as_bytes(), false);
            assert_eq!(result, input.1, "Failed for '{:?}", input.0);
        }
    }
}

static ENTITY_HASH: &[u32; 260] = &[
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 0, 35, 30, 0, 20, 75,
    18080, 50, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 360, 660, 1375, 1590, 165,
    897, 395, 1270, 1990, 80, 365, 515, 695, 630, 402, 700, 195, 765, 125, 900, 1210, 570, 50, 415,
    495, 150, 18080, 15, 18080, 0, 18080, 18080, 4491, 45, 2786, 130, 15, 535, 2358, 3013, 270,
    1125, 3797, 5, 155, 100, 1440, 75, 805, 0, 10, 20, 35, 305, 1730, 3448, 4111, 145, 110, 420, 0,
    0, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080, 18080,
    18080, 18080, 18080,
];

static ENTITY_MAP: &[u32; 18016] = &[
    0x02192, 0, 0, 0, 0, 0x02190, 0, 0, 0, 0, 0, 0, 0, 0, 0x02AAB, 0, 0, 0, 0, 0, 0, 0, 0, 0x0226A,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A99, 0, 0x02191, 0, 0, 0, 0x02225, 0x00029, 0x021FE, 0, 0, 0,
    0x00028, 0x021FD, 0x021A3, 0, 0x022A5, 0x02225, 0, 0x021A2, 0, 0, 0x022D5, 0, 0, 0, 0x02A95,
    0x02002, 0, 0, 0, 0, 0, 0, 0, 0, 0x02280, 0, 0x021BE, 0, 0, 0, 0x02A89, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x02591, 0x029E3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000AC, 0, 0x021C4, 0,
    0, 0, 0, 0x021C7, 0, 0, 0, 0, 0x02190, 0, 0, 0x0002F, 0, 0, 0, 0, 0, 0, 0, 0x02034, 0x02A35, 0,
    0x02193, 0x02592, 0, 0x02A34, 0x002D9, 0, 0x021AE, 0, 0, 0, 0x02226, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x021D2, 0x02199, 0x02035, 0, 0, 0x021D0, 0, 0, 0, 0, 0, 0x02593, 0, 0, 0, 0x02025, 0x021C2, 0,
    0, 0x0226E, 0, 0x021C1, 0, 0, 0, 0x029C4, 0x021BD, 0x0233F, 0, 0, 0, 0, 0, 0, 0, 0x021D1, 0, 0,
    0x02213, 0x02113, 0, 0x02AEE, 0, 0, 0, 0, 0x021DB, 0x02975, 0, 0, 0, 0x021DA, 0, 0, 0, 0x02226,
    0x0219A, 0, 0, 0x025CA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02026, 0x021BF, 0, 0, 0x022FC, 0,
    0x022E6, 0, 0x02264, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02147, 0, 0, 0x02196, 0x02993, 0, 0x02A7D, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0222B, 0, 0, 0, 0, 0x0200F, 0x003F5,
    0x02242, 0, 0, 0, 0x000AF, 0, 0, 0, 0x0231E, 0x021D3, 0, 0x02133, 0, 0, 0, 0, 0, 0x02972, 0,
    0x003C5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02195, 0, 0x021AC, 0x02208, 0x0231C, 0,
    0x02198, 0x021AB, 0, 0x003C4, 0, 0x021C3, 0, 0x0227A, 0, 0, 0, 0, 0x02260, 0x025A1, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0x021C9, 0, 0x02214, 0, 0, 0x021C6, 0x0296C, 0, 0, 0, 0x02192, 0x0296A, 0,
    0, 0, 0x02971, 0, 0, 0x02225, 0, 0, 0, 0, 0x0223C, 0, 0, 0, 0, 0, 0, 0x021C0, 0, 0, 0, 0,
    0x021BC, 0, 0, 0, 0, 0x02666, 0, 0, 0, 0x022A5, 0x021C8, 0, 0, 0, 0x02ADB, 0x02A3C, 0, 0, 0, 0,
    0, 0x0231C, 0, 0x02223, 0, 0x02274, 0, 0, 0, 0, 0x02197, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021CC, 0, 0x02A24, 0, 0, 0, 0x02929, 0, 0, 0, 0x0219B, 0,
    0x003C0, 0, 0, 0, 0x0230F, 0, 0x02A53, 0, 0, 0, 0, 0, 0x02AB7, 0, 0, 0, 0, 0x021D5, 0, 0x0002A,
    0, 0, 0, 0, 0x025A1, 0x0220B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0221D, 0x02253,
    0, 0, 0, 0, 0, 0x0231E, 0, 0, 0, 0, 0, 0, 0x02A87, 0, 0, 0, 0, 0, 0x022C4, 0, 0, 0, 0x0200E, 0,
    0x02AFD, 0, 0, 0x0231F, 0, 0, 0x02105, 0, 0, 0x0219E, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0x022D8, 0, 0x02964, 0, 0x0230D, 0, 0x0231D, 0x02962, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0290D, 0, 0, 0, 0x02963, 0x0290C, 0, 0, 0, 0, 0,
    0x025A1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0007C, 0, 0, 0, 0, 0, 0, 0, 0x0003B, 0, 0, 0, 0, 0,
    0x0227E, 0x0003D, 0x02974, 0, 0, 0, 0, 0x02973, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0231D,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02148, 0x02AEC, 0, 0, 0x0002E, 0, 0, 0, 0, 0, 0, 0, 0x02A54, 0,
    0, 0, 0, 0, 0, 0, 0, 0x021AB, 0, 0, 0, 0, 0x02223, 0, 0x021CB, 0, 0, 0x02270, 0x02965, 0, 0, 0,
    0, 0, 0, 0x0230E, 0, 0, 0, 0, 0, 0, 0, 0x000A0, 0, 0, 0x02990, 0, 0, 0, 0, 0x0298D, 0, 0, 0,
    0x0296B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0002B, 0, 0, 0, 0, 0, 0, 0x0231F,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02292, 0, 0, 0x02291, 0, 0, 0x02A75, 0, 0, 0, 0x02290, 0,
    0, 0x0228F, 0, 0, 0, 0, 0, 0, 0, 0x021CE, 0, 0, 0, 0, 0, 0x025AE, 0, 0, 0, 0, 0, 0, 0, 0x021A0,
    0x021D9, 0x0230C, 0, 0, 0, 0, 0, 0, 0x000A8, 0, 0, 0x0007D, 0x0039E, 0, 0, 0, 0x0007B, 0x022C4,
    0, 0, 0, 0x02A25, 0, 0, 0, 0, 0x02200, 0, 0x025B4, 0, 0, 0, 0, 0, 0, 0, 0, 0x025C2, 0x025B8, 0,
    0, 0, 0, 0x02AAA, 0, 0, 0x02916, 0, 0, 0, 0, 0x02945, 0, 0, 0, 0x021CD, 0x02939, 0, 0, 0, 0,
    0x02005, 0, 0, 0, 0, 0x003DC, 0x0223D, 0, 0, 0, 0x022BA, 0, 0x02A85, 0, 0, 0x02235, 0, 0, 0, 0,
    0, 0, 0x022A4, 0, 0, 0x02016, 0, 0, 0, 0x021D6, 0, 0, 0, 0, 0, 0, 0x02235, 0, 0x02003, 0, 0, 0,
    0x02AAF, 0x029EB, 0, 0, 0x0298E, 0, 0, 0, 0, 0x0298F, 0, 0, 0, 0, 0x025BE, 0, 0, 0, 0x02004, 0,
    0, 0, 0x00149, 0, 0, 0, 0, 0x02986, 0, 0, 0, 0, 0x02985, 0, 0, 0, 0, 0, 0x02336, 0, 0, 0, 0,
    0x02057, 0, 0, 0, 0x0266F, 0, 0, 0, 0, 0, 0x0296D, 0, 0, 0, 0x021D8, 0, 0x0290F, 0, 0, 0, 0,
    0x0290E, 0x02249, 0, 0, 0, 0, 0x000A8, 0, 0, 0x022E0, 0x02ABB, 0, 0, 0, 0x02666, 0, 0, 0, 0,
    0x02305, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000B1, 0, 0, 0, 0, 0, 0,
    0, 0x02248, 0, 0, 0, 0, 0x0226E, 0, 0x0211C, 0x021A6, 0, 0x02AF2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x000B1, 0, 0x02A0D, 0, 0x02268, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021D7, 0, 0, 0x02264, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02266, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021CF,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003A0, 0x02228, 0, 0, 0, 0, 0, 0, 0, 0x02234, 0, 0x1D52F, 0,
    0, 0, 0, 0x1D529, 0, 0, 0, 0, 0x1D530, 0, 0, 0, 0, 0x1D522, 0, 0, 0x003C6, 0, 0x1D531, 0x025AA,
    0, 0, 0, 0, 0, 0, 0, 0x02A22, 0, 0, 0, 0, 0, 0x1D532, 0, 0, 0x022BB, 0, 0, 0, 0, 0x021CC, 0,
    0x1D51F, 0, 0, 0, 0, 0x1D51A, 0x02AEB, 0x02293, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0298C, 0, 0, 0, 0,
    0x0298B, 0, 0x02272, 0, 0, 0, 0, 0, 0x1D52D, 0, 0, 0, 0, 0x1D50D, 0, 0x0221D, 0, 0, 0, 0, 0,
    0x02640, 0, 0x02221, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D52B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0226C, 0, 0, 0, 0, 0, 0x003A4, 0, 0, 0, 0, 0x1D516, 0, 0, 0, 0, 0x1D521, 0, 0, 0x022B0, 0,
    0x02312, 0x0219F, 0, 0, 0, 0, 0, 0, 0x0212C, 0, 0x1D537, 0, 0, 0x02AE8, 0, 0x02128, 0, 0, 0, 0,
    0x1D52A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D508, 0, 0, 0, 0x0221D, 0, 0, 0, 0, 0, 0, 0, 0x02A72, 0,
    0, 0, 0, 0, 0x0291A, 0, 0x02313, 0, 0, 0x02919, 0, 0, 0, 0, 0, 0, 0x1D514, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x0005E, 0, 0x000A3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02044, 0, 0, 0, 0, 0, 0x0230B, 0, 0, 0,
    0x02252, 0x0230A, 0x02A26, 0, 0x02A87, 0x02268, 0, 0, 0, 0, 0, 0, 0, 0x02A12, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x02035, 0, 0, 0, 0x022C1, 0, 0x02237, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022EC, 0,
    0x1D526, 0, 0, 0, 0, 0, 0, 0x00113, 0, 0, 0, 0x00192, 0, 0, 0, 0, 0, 0x0226C, 0x000BE, 0,
    0x029E4, 0, 0, 0, 0, 0, 0, 0x0016B, 0, 0, 0, 0, 0, 0, 0, 0x1D533, 0, 0x02AB3, 0x02277, 0, 0, 0,
    0, 0, 0, 0x00026, 0, 0, 0, 0, 0, 0, 0, 0x000BC, 0x02273, 0, 0x02AA2, 0, 0x02157, 0, 0, 0, 0,
    0x0291C, 0x02265, 0, 0, 0, 0x0291B, 0, 0x003C6, 0, 0, 0x022DB, 0, 0, 0, 0x025B1, 0x02158, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0x02156, 0, 0x1D504, 0, 0, 0x02155, 0, 0x1D50E, 0, 0, 0, 0, 0, 0, 0x02A77,
    0, 0, 0, 0, 0x0260E, 0x02154, 0, 0, 0, 0, 0x02153, 0, 0x003C1, 0, 0, 0x0215C, 0, 0x02238,
    0x003D2, 0, 0x000BD, 0x02208, 0x1D50A, 0, 0, 0, 0, 0, 0x02270, 0x1D512, 0, 0x02235, 0, 0x029B7,
    0, 0x025AA, 0, 0, 0x022CD, 0x025FA, 0, 0, 0x1D51B, 0, 0, 0, 0, 0, 0, 0, 0x0215B, 0, 0, 0,
    0x00112, 0x025AA, 0, 0, 0, 0, 0x02030, 0, 0x029C0, 0, 0x02236, 0x0215E, 0, 0, 0, 0x025F8, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0x02A33, 0, 0x02905, 0, 0, 0, 0, 0, 0, 0, 0x0215D, 0, 0, 0, 0, 0x02267,
    0, 0, 0, 0x021BA, 0x02159, 0, 0, 0, 0, 0, 0x024C8, 0, 0, 0, 0x0210F, 0, 0, 0x1D563, 0x02246, 0,
    0, 0, 0x1D55D, 0, 0x02306, 0, 0x1D51C, 0x1D564, 0, 0x022ED, 0, 0, 0x1D556, 0, 0, 0, 0, 0x1D565,
    0x022EA, 0, 0, 0, 0x0226E, 0x0223A, 0x0215A, 0, 0x1D50F, 0x021A1, 0, 0, 0, 0x000A8, 0x1D566, 0,
    0, 0, 0, 0, 0x029BE, 0, 0, 0x02216, 0x1D553, 0x0012B, 0, 0, 0x1D523, 0x1D54E, 0, 0, 0x02A39,
    0x02127, 0x02274, 0x000FE, 0x02278, 0, 0, 0, 0x02A96, 0, 0, 0, 0x02270, 0, 0, 0, 0, 0, 0x02209,
    0, 0, 0, 0x1D561, 0, 0, 0, 0x02126, 0x1D541, 0, 0x02A95, 0, 0x1D519, 0, 0, 0, 0x0003E, 0,
    0x02A9D, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D55F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x003D6, 0, 0, 0, 0, 0, 0, 0x02228, 0, 0x1D54A, 0, 0x02316, 0, 0, 0x1D555, 0, 0x0292A, 0x02910,
    0, 0, 0x00100, 0, 0, 0, 0x0211C, 0, 0, 0, 0x1D511, 0x1D56B, 0, 0, 0, 0, 0x02124, 0, 0x0223C, 0,
    0, 0x1D55E, 0x022BF, 0, 0, 0, 0, 0, 0x00024, 0, 0, 0x1D53C, 0x0211D, 0, 0x029BB, 0, 0x0005D, 0,
    0, 0, 0x1D505, 0x0005B, 0, 0, 0x0014C, 0, 0, 0, 0, 0, 0, 0, 0x025F9, 0, 0, 0, 0, 0x02226, 0, 0,
    0, 0x0211A, 0x02250, 0, 0, 0, 0x02288, 0, 0, 0x02118, 0, 0, 0, 0, 0x0003C, 0x1D510, 0, 0, 0, 0,
    0x1D513, 0, 0x021BB, 0x02289, 0, 0, 0, 0, 0x02AF3, 0, 0x02AA1, 0, 0x022B2, 0, 0, 0, 0x02202,
    0x022F2, 0, 0, 0x029E5, 0x02A56, 0, 0x02903, 0x003B5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02310,
    0x022EB, 0, 0, 0, 0, 0, 0, 0x003C5, 0, 0, 0, 0, 0, 0, 0, 0x0212C, 0, 0, 0x02A5D, 0x1D55A, 0, 0,
    0, 0, 0x1D4C7, 0, 0, 0, 0x0211C, 0x1D4C1, 0, 0, 0, 0, 0x1D4C8, 0, 0, 0, 0, 0x0212F, 0, 0, 0, 0,
    0x1D4C9, 0, 0, 0, 0, 0x0221A, 0x02203, 0, 0, 0, 0x1D567, 0, 0, 0, 0, 0x1D4CA, 0x02216, 0, 0, 0,
    0, 0, 0, 0, 0x1D52E, 0x1D4B7, 0, 0, 0, 0, 0x1D4B2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022B7, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D4C5, 0, 0, 0, 0, 0x1D4A5, 0, 0x02A93, 0x02255, 0, 0x1D538,
    0, 0, 0, 0, 0x1D542, 0, 0x02AF0, 0x022F7, 0, 0x000BA, 0x000B5, 0, 0, 0, 0x1D4C3, 0x02242, 0, 0,
    0, 0, 0, 0x02902, 0, 0x02241, 0, 0, 0, 0x00395, 0, 0x00021, 0x022C0, 0, 0, 0, 0x1D53E, 0, 0, 0,
    0x02207, 0x1D4AE, 0, 0x1D546, 0, 0, 0x1D4B9, 0, 0x0223E, 0x02A3C, 0, 0, 0x1D509, 0x02323,
    0x02240, 0x1D517, 0x1D54F, 0, 0, 0, 0, 0x1D4CF, 0, 0, 0x02247, 0x0222C, 0x1D4B5, 0, 0, 0, 0,
    0x1D4C2, 0, 0, 0x0227E, 0, 0, 0, 0, 0, 0, 0x02130, 0, 0, 0, 0, 0, 0, 0x000F7, 0, 0, 0, 0,
    0x02203, 0, 0, 0x02720, 0x022B3, 0x02A5F, 0, 0, 0, 0x02224, 0, 0, 0x023E7, 0, 0, 0, 0, 0,
    0x1D4AC, 0x02A2F, 0, 0, 0, 0, 0x003B5, 0, 0x022D6, 0x003A6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0x1D550, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02009, 0, 0x0223C, 0, 0x02032, 0, 0, 0,
    0x1D543, 0, 0x02119, 0, 0, 0, 0, 0x000A6, 0, 0, 0, 0, 0x020DC, 0, 0, 0x025AD, 0, 0, 0, 0,
    0x1D557, 0, 0, 0x02266, 0, 0x000A7, 0, 0, 0, 0, 0x1D4BE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D54D, 0, 0, 0, 0, 0x02284, 0, 0, 0, 0, 0x1D4CB, 0, 0, 0, 0,
    0x000A9, 0, 0x022A5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02285, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02115, 0, 0, 0, 0, 0x1D49C, 0, 0, 0,
    0x1D527, 0x1D4A6, 0, 0, 0, 0x003D6, 0, 0, 0, 0, 0x022D2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D539,
    0, 0, 0, 0x003A1, 0, 0, 0, 0x029EB, 0x0220B, 0x1D4A2, 0, 0, 0, 0, 0, 0, 0x1D4AA, 0, 0, 0, 0, 0,
    0x022C4, 0, 0, 0, 0, 0, 0, 0x1D4B3, 0, 0, 0, 0, 0x1D544, 0x02209, 0x02A6E, 0, 0x000F7, 0x02119,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0007C, 0, 0, 0, 0, 0x00022, 0, 0, 0, 0x1D518, 0,
    0x0220C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00025, 0,
    0, 0, 0, 0x02904, 0, 0, 0, 0, 0, 0, 0, 0x0211D, 0, 0, 0, 0, 0x1D4B4, 0, 0, 0, 0, 0, 0x02261, 0,
    0, 0, 0x000B1, 0, 0, 0, 0x0210C, 0x0266D, 0, 0, 0, 0x02249, 0x02112, 0x0222D, 0, 0, 0, 0, 0,
    0x02276, 0, 0, 0x02224, 0, 0, 0, 0, 0x1D562, 0, 0, 0, 0, 0x1D4BB, 0, 0, 0, 0, 0, 0x02277, 0, 0,
    0, 0x0291E, 0, 0x0227C, 0, 0x02A15, 0x0291D, 0, 0, 0, 0, 0, 0, 0, 0x02AA0, 0, 0, 0, 0x003F6, 0,
    0, 0, 0, 0, 0, 0, 0x1D4B1, 0, 0, 0, 0, 0, 0, 0, 0x022FE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0x02A8A, 0, 0x02210, 0, 0, 0, 0, 0, 0, 0x02AF1, 0x02AB8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0212D, 0, 0, 0, 0, 0, 0, 0, 0x1D53D, 0x022E2, 0, 0x1D54B, 0, 0, 0, 0, 0x1D4A9, 0, 0, 0, 0, 0,
    0, 0, 0x021AC, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022E3, 0x021BF, 0, 0, 0, 0, 0,
    0x0212C, 0, 0, 0x022EC, 0, 0x02306, 0, 0, 0x022EA, 0, 0x000AA, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D52C,
    0, 0, 0, 0, 0, 0, 0, 0x02A2E, 0, 0, 0x02A97, 0, 0x02A2D, 0x02A43, 0, 0x02133, 0, 0, 0, 0x0014A,
    0x1D4AB, 0, 0, 0, 0, 0x02016, 0x02322, 0, 0, 0, 0, 0x0016A, 0, 0, 0, 0, 0, 0, 0, 0, 0x02272, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02720, 0, 0x02AED, 0, 0, 0, 0x022E7, 0,
    0x02265, 0, 0, 0, 0, 0, 0, 0, 0x0227F, 0, 0, 0, 0, 0, 0, 0, 0x02A7E, 0, 0, 0x0211B, 0, 0, 0,
    0x02AE4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02200, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02AD9, 0, 0, 0, 0, 0x02289, 0, 0, 0, 0x1D4C6, 0x02288, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x029BF, 0,
    0, 0, 0x00159, 0, 0, 0, 0, 0x0013E, 0, 0, 0x0007D, 0x1D507, 0x00161, 0x02992, 0x02213, 0x0007B,
    0, 0x0011B, 0x02991, 0, 0, 0, 0x00165, 0, 0, 0, 0, 0x02A6A, 0, 0, 0, 0, 0x1D55B, 0, 0, 0,
    0x02293, 0, 0x02A8B, 0, 0, 0, 0, 0x02033, 0, 0, 0, 0, 0, 0, 0x022FA, 0, 0, 0, 0x027E9, 0x02A9F,
    0x0227A, 0x02A7F, 0, 0x027E8, 0, 0x003A7, 0, 0x02226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022DB, 0,
    0x02110, 0, 0, 0x02131, 0x0220C, 0, 0x1D4AF, 0x02111, 0, 0, 0x02A3A, 0x000B7, 0, 0, 0, 0x02216,
    0, 0x021A4, 0, 0, 0, 0, 0, 0, 0, 0, 0x00148, 0, 0, 0, 0, 0x00131, 0x003F6, 0, 0, 0, 0, 0, 0, 0,
    0, 0x1D54C, 0x02AB9, 0x022E1, 0, 0, 0, 0x0014D, 0, 0, 0, 0x00160, 0, 0, 0, 0, 0x0010F, 0, 0,
    0x02A42, 0, 0, 0, 0, 0, 0, 0x003DD, 0, 0, 0, 0, 0x0017E, 0, 0, 0, 0x1D534, 0x0017D, 0, 0x0297D,
    0, 0x021A5, 0, 0, 0x0297C, 0, 0, 0, 0, 0, 0, 0x0227C, 0x0011A, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0210D, 0, 0x021E5, 0, 0, 0, 0, 0x021E4, 0, 0, 0, 0, 0x0297E, 0x02A14, 0, 0x02190, 0, 0x02A88,
    0, 0, 0, 0x029B9, 0, 0, 0, 0, 0, 0x02AB0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0296E, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0x0299C, 0, 0, 0, 0, 0, 0x0220F, 0, 0, 0, 0x003D1, 0x02111, 0, 0x025CB, 0, 0x0221A, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02295, 0, 0, 0, 0x0200C, 0, 0x02933, 0, 0,
    0, 0x02201, 0, 0, 0, 0, 0, 0x022C6, 0, 0, 0x02660, 0, 0x02296, 0, 0, 0x02102, 0, 0x0297F, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02289, 0, 0x02A74, 0x0211B, 0, 0x02288, 0, 0, 0, 0, 0x1D4BF, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A0C, 0x02660, 0, 0, 0, 0, 0, 0, 0, 0x021AE, 0,
    0, 0, 0, 0, 0, 0x003A5, 0, 0, 0, 0, 0x00040, 0x02309, 0x1D560, 0, 0, 0, 0x02308, 0, 0, 0, 0, 0,
    0, 0x02192, 0, 0x02194, 0, 0, 0, 0, 0, 0, 0, 0x02237, 0, 0x02AAD, 0, 0, 0, 0, 0, 0, 0x02272,
    0x021A7, 0, 0, 0, 0x02015, 0, 0, 0, 0, 0x02242, 0, 0, 0, 0, 0x1D4B0, 0, 0, 0, 0, 0x02251, 0, 0,
    0, 0x021FF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02111, 0x02234, 0, 0, 0, 0, 0x0223D, 0x029A5, 0, 0, 0,
    0, 0, 0, 0, 0x021C5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02994, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x0210B, 0, 0, 0, 0, 0, 0, 0, 0x029DA, 0, 0, 0, 0x02297, 0, 0x0227E, 0, 0, 0x021B5, 0, 0,
    0x02241, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003C8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D53B,
    0, 0x02663, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021D4, 0, 0x0229B, 0, 0, 0, 0, 0x0013D, 0, 0, 0,
    0x021CA, 0, 0, 0x000B8, 0, 0x002DC, 0, 0, 0, 0, 0, 0x02500, 0, 0, 0, 0, 0x02A73, 0, 0x02A86, 0,
    0, 0x02AB5, 0, 0x02305, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022DA, 0x1D49E, 0x02146, 0, 0, 0, 0,
    0, 0x02201, 0, 0x02122, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02243, 0, 0x0005C, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0221D, 0, 0, 0, 0, 0x02112, 0, 0, 0x02AE9, 0x000D7, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x022E8, 0x0210B, 0, 0, 0, 0, 0x02134, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00147, 0, 0,
    0, 0, 0x1D568, 0, 0, 0x02642, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003D2, 0x0222D, 0, 0, 0, 0,
    0x022A7, 0, 0, 0, 0, 0, 0, 0x02294, 0, 0, 0x029F6, 0, 0x0221D, 0, 0, 0, 0x0012A, 0, 0x02A57, 0,
    0, 0, 0, 0, 0, 0, 0x021DD, 0, 0, 0, 0x022A0, 0, 0, 0, 0x02920, 0, 0, 0x0226D, 0, 0x0291F, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0x02269, 0x02665, 0, 0, 0, 0, 0, 0, 0, 0, 0x02AB4, 0x0225F, 0, 0x02265, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022EE, 0, 0, 0x02928, 0, 0, 0x02267, 0, 0, 0, 0, 0,
    0x0296E, 0, 0, 0x029DC, 0x02131, 0, 0, 0, 0, 0x02665, 0, 0, 0x0003C, 0, 0, 0, 0, 0x02948, 0, 0,
    0, 0x00158, 0, 0, 0, 0, 0x1D49F, 0x02A8D, 0, 0, 0, 0, 0, 0x1D524, 0, 0x02A92, 0, 0, 0, 0, 0, 0,
    0, 0x02717, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003F1, 0, 0, 0, 0, 0, 0, 0x022D6,
    0x0005F, 0x022CD, 0, 0, 0x02A00, 0x02663, 0, 0, 0x02256, 0, 0, 0, 0, 0x02606, 0, 0, 0, 0x0222F,
    0, 0x027F7, 0, 0, 0, 0, 0x02205, 0, 0, 0, 0, 0x022F0, 0, 0, 0, 0, 0, 0, 0, 0x022E8, 0, 0, 0, 0,
    0, 0, 0, 0x02A30, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02244, 0, 0, 0, 0, 0, 0, 0x02122, 0, 0, 0,
    0x0229E, 0, 0, 0x025FC, 0, 0x02AAF, 0, 0x02A79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0266E, 0x1D540, 0,
    0, 0, 0x027F5, 0x00164, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D4CC, 0, 0, 0x0210F, 0, 0x02A02, 0, 0, 0,
    0, 0x02996, 0x02276, 0, 0, 0, 0, 0, 0, 0x02280, 0, 0, 0, 0, 0, 0x022F1, 0, 0, 0, 0x022C6, 0,
    0x02217, 0, 0x02214, 0, 0x029C2, 0x00155, 0, 0, 0, 0, 0x0013A, 0, 0, 0, 0x02A71, 0x0015B,
    0x021CE, 0, 0, 0, 0x000E9, 0, 0, 0, 0, 0x02229, 0, 0, 0, 0, 0x000BF, 0, 0, 0x02A02, 0x0228E, 0,
    0x02227, 0, 0, 0, 0x000FA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000A8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0x02A88, 0x02269, 0, 0, 0x02A13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x025B9, 0, 0, 0x02062, 0,
    0x025C3, 0, 0, 0, 0, 0, 0, 0x00009, 0, 0x02AA8, 0x02A9E, 0, 0x02976, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x00144, 0, 0x02128, 0, 0, 0, 0, 0, 0x025B5, 0, 0, 0, 0, 0x022E0, 0, 0, 0, 0, 0, 0x02A5B, 0,
    0x0204F, 0, 0, 0, 0x0015A, 0, 0, 0, 0, 0, 0, 0, 0, 0x022C7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0017A, 0, 0, 0, 0, 0x00179, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003A8, 0x000C9, 0, 0, 0,
    0, 0, 0, 0x0229B, 0, 0, 0, 0, 0, 0, 0x000D0, 0, 0x02A96, 0, 0, 0x027F6, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0x02299, 0x0226B, 0, 0x025BF, 0, 0x02A23, 0x0211E, 0x002DD, 0, 0, 0,
    0x003BE, 0, 0, 0, 0x0226A, 0x022C3, 0, 0x02AAC, 0, 0, 0, 0, 0, 0, 0x02110, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003C2, 0, 0x029CE, 0, 0, 0x024C8, 0, 0, 0,
    0, 0x1D520, 0, 0, 0, 0, 0x025EC, 0, 0, 0, 0, 0x000ED, 0x02323, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x022C2, 0, 0, 0, 0, 0, 0x0226F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022F4, 0, 0, 0, 0, 0,
    0x0226F, 0, 0, 0, 0, 0, 0x02210, 0, 0, 0, 0, 0x1D558, 0, 0, 0x0226A, 0, 0, 0, 0, 0, 0, 0x000BD,
    0, 0, 0, 0, 0, 0x02250, 0, 0x0203E, 0, 0, 0, 0, 0, 0, 0, 0, 0x02234, 0, 0, 0, 0, 0, 0, 0, 0,
    0x022B5, 0, 0, 0, 0, 0x022B4, 0x000C1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02208, 0, 0, 0, 0, 0x02261,
    0, 0, 0, 0, 0, 0, 0, 0x02117, 0, 0, 0x029A7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x000D3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A01, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02243, 0, 0, 0,
    0, 0, 0x0010C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022B9, 0, 0, 0, 0, 0, 0, 0x02205, 0,
    0, 0, 0, 0, 0, 0x0223C, 0, 0, 0, 0, 0, 0x020DB, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x029A6, 0, 0, 0, 0, 0, 0x1D525, 0, 0, 0x02A0C, 0, 0, 0, 0, 0x000DD, 0, 0, 0, 0x027FA, 0, 0, 0,
    0, 0, 0x02255, 0, 0, 0x02262, 0, 0, 0, 0, 0, 0x0221E, 0x00139, 0, 0, 0x02275, 0, 0, 0, 0, 0, 0,
    0x003C7, 0, 0, 0, 0, 0, 0, 0x02A83, 0, 0, 0, 0, 0x021A6, 0, 0, 0, 0, 0, 0, 0, 0x022C3, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x02AE6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x027F8, 0, 0, 0, 0, 0, 0,
    0x0227A, 0x021C8, 0, 0, 0, 0, 0x02218, 0, 0x003BD, 0, 0, 0, 0, 0, 0, 0, 0, 0x0210A, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000AD, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x021D4, 0, 0, 0x00143, 0, 0, 0, 0, 0, 0, 0, 0, 0x003BC, 0, 0, 0, 0x022DA, 0x000AF, 0x0010E, 0,
    0x02216, 0, 0, 0, 0, 0x0222F, 0, 0, 0, 0, 0, 0, 0, 0x022CC, 0, 0x0224C, 0, 0, 0x022CB, 0, 0, 0,
    0, 0, 0, 0x021F5, 0, 0, 0, 0, 0, 0x02A4B, 0, 0, 0x0224E, 0, 0x02970, 0, 0x02133, 0, 0, 0, 0,
    0x02292, 0, 0x002C7, 0, 0, 0, 0, 0, 0x000DE, 0, 0x0200A, 0, 0, 0, 0, 0, 0x021D0, 0x02290,
    0x021BE, 0, 0x003C2, 0, 0x029B5, 0, 0, 0x02211, 0, 0, 0x0205F, 0, 0, 0, 0x029DD, 0x027FC, 0, 0,
    0x029C3, 0, 0, 0x0229E, 0x023B1, 0, 0, 0, 0x02290, 0x023B0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02949,
    0x1D554, 0, 0, 0, 0x00140, 0, 0, 0x02AEF, 0, 0, 0, 0, 0, 0x000AE, 0, 0, 0, 0x02254, 0, 0x00154,
    0, 0, 0x02223, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02271, 0, 0x02145, 0, 0, 0,
    0x022D4, 0, 0, 0, 0, 0, 0, 0x027F9, 0, 0, 0x023B4, 0, 0, 0x00023, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x0225C, 0, 0, 0, 0, 0, 0, 0, 0, 0x023B5, 0, 0, 0x02211, 0, 0x0003A, 0x02A3B, 0, 0x0228E,
    0x003F1, 0, 0, 0, 0, 0, 0x02296, 0, 0x02294, 0, 0, 0x02A8C, 0, 0x0227C, 0, 0x02267, 0x0233D, 0,
    0, 0, 0, 0, 0x02224, 0x02AB5, 0x0039F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02588, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02204, 0x02204, 0, 0, 0, 0, 0, 0x000AE, 0, 0, 0, 0, 0, 0,
    0x022F3, 0, 0, 0, 0, 0, 0, 0, 0, 0x023B6, 0x02553, 0, 0, 0, 0, 0, 0, 0x00119, 0, 0, 0x1D535, 0,
    0x02248, 0, 0, 0x029CD, 0, 0, 0, 0, 0x02559, 0, 0, 0, 0, 0, 0x02A47, 0x00173, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D559,
    0x003B5, 0x02227, 0, 0, 0, 0, 0, 0x021D4, 0, 0, 0, 0x02A94, 0, 0, 0, 0, 0, 0, 0x0296F, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02929, 0x0255F, 0, 0, 0, 0x00448, 0x02564, 0, 0,
    0, 0x1D4B8, 0, 0, 0, 0, 0, 0, 0, 0, 0x02298, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x02927, 0x02A00, 0x021A2, 0, 0, 0, 0, 0x023E2, 0, 0, 0, 0x000B6, 0x02556, 0, 0, 0, 0, 0,
    0, 0x00118, 0, 0, 0, 0, 0, 0, 0x0211A, 0, 0, 0, 0, 0, 0x0255C, 0, 0, 0x02281, 0, 0, 0x022BD, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02225, 0, 0, 0, 0, 0, 0x00459, 0x025FB, 0, 0, 0, 0, 0, 0x02423,
    0x0039D, 0, 0, 0, 0, 0, 0, 0, 0x02251, 0, 0x02134, 0x0222E, 0, 0, 0, 0, 0, 0x0045F, 0, 0, 0,
    0x02322, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00436, 0, 0, 0, 0, 0, 0x0224F, 0,
    0x02ABA, 0x02020, 0, 0x02562, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0012F, 0, 0, 0, 0, 0x0039C, 0,
    0, 0, 0, 0, 0, 0, 0, 0x022C5, 0, 0, 0, 0, 0x00117, 0, 0, 0, 0, 0x020DB, 0, 0x02232, 0x0045A, 0,
    0x02A91, 0, 0, 0, 0, 0, 0, 0x000DA, 0, 0x02567, 0, 0, 0, 0, 0, 0, 0, 0, 0x02ACF, 0x02231, 0, 0,
    0, 0, 0, 0x022FB, 0, 0x029BC, 0x00452, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00027, 0, 0, 0, 0, 0x02AD0,
    0, 0x1D4BD, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0013F, 0, 0, 0, 0x00104, 0, 0x000B9, 0, 0,
    0x0225A, 0, 0x1D528, 0, 0, 0, 0, 0x003B9, 0, 0x027E9, 0, 0, 0x00451, 0, 0x027E8, 0x023DF, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0229A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0200B, 0, 0, 0x003B2, 0,
    0, 0, 0, 0, 0, 0, 0x000BB, 0, 0x000B2, 0x0203A, 0x0017C, 0x000AB, 0, 0, 0x02039, 0x0017B, 0, 0,
    0, 0, 0x022F9, 0, 0, 0, 0, 0, 0, 0x02283, 0, 0, 0x00116, 0, 0, 0x029DE, 0, 0, 0, 0, 0, 0x021B1,
    0, 0, 0, 0, 0x021B0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022A1, 0, 0x02191, 0, 0, 0x000B3, 0, 0, 0,
    0x00106, 0, 0, 0, 0x001B5, 0, 0, 0, 0x0201C, 0, 0, 0, 0x02018, 0x02191, 0, 0, 0, 0, 0, 0,
    0x02A10, 0, 0, 0, 0, 0x0224A, 0, 0, 0, 0, 0, 0, 0, 0x1D569, 0, 0, 0x003B6, 0, 0, 0, 0, 0x00396,
    0, 0x002DA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02243, 0, 0, 0, 0, 0x02295, 0, 0, 0, 0, 0, 0x000F3,
    0x02AE7, 0x0290D, 0, 0, 0, 0, 0, 0, 0, 0, 0x02912, 0, 0x02199, 0, 0x022D1, 0, 0, 0, 0, 0x003F0,
    0, 0x00393, 0x02A80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02062, 0, 0, 0, 0, 0, 0, 0x022B6, 0, 0,
    0, 0, 0, 0, 0x02AC0, 0, 0, 0x0227B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0223B, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x0200D, 0x02244, 0, 0, 0, 0, 0, 0, 0x029B0, 0, 0x02928, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x00435, 0, 0x02196, 0, 0x022CA, 0, 0, 0, 0x022A8, 0x022C9, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02212, 0x02A4D, 0x02551, 0, 0x00120, 0, 0, 0, 0, 0,
    0x0219D, 0, 0, 0, 0x02AB6, 0, 0x1D536, 0, 0, 0, 0, 0x02245, 0, 0, 0, 0, 0x02019, 0, 0x02271, 0,
    0, 0x0201A, 0, 0, 0, 0, 0, 0, 0x02ABC, 0, 0x029B1, 0x0232E, 0x02061, 0, 0x02198, 0, 0, 0,
    0x0226F, 0, 0, 0, 0, 0, 0, 0x003F0, 0, 0, 0x02259, 0, 0, 0, 0x029C1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x00428, 0, 0, 0, 0, 0x02279, 0, 0, 0x02248, 0x0229F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0042F, 0,
    0, 0, 0, 0x00416, 0, 0, 0x02AAE, 0x02254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D4CD, 0, 0, 0,
    0x000A1, 0, 0, 0x02A7D, 0, 0, 0, 0x02AC2, 0x0016D, 0, 0, 0, 0, 0, 0, 0, 0x02197, 0, 0, 0, 0, 0,
    0, 0, 0, 0x0045E, 0, 0, 0, 0, 0x022B8, 0, 0x02A66, 0, 0, 0x02AD2, 0, 0, 0x0014B, 0, 0x02AD1, 0,
    0x0227F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x025B5, 0, 0, 0x02193, 0, 0, 0x02A27, 0, 0, 0,
    0x02063, 0, 0, 0, 0, 0x1D55C, 0, 0, 0, 0x02A2A, 0, 0, 0, 0, 0, 0, 0x022F5, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x02A44, 0, 0, 0, 0, 0x02273, 0, 0, 0x025C3, 0, 0, 0x02A90, 0, 0, 0x02554, 0x027EA, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02218, 0, 0, 0, 0, 0, 0x0255A, 0, 0x022AB, 0, 0, 0, 0, 0x02225,
    0x02238, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A4C, 0, 0, 0, 0, 0, 0, 0x02007, 0, 0, 0, 0x0010D, 0, 0,
    0, 0, 0, 0, 0, 0x0201A, 0, 0, 0, 0, 0, 0x0229F, 0, 0, 0, 0, 0, 0x021D3, 0, 0, 0, 0, 0, 0x021B0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00425, 0x000DF, 0, 0, 0, 0, 0x002D8, 0, 0, 0x022E9,
    0x022C2, 0x02560, 0, 0, 0x02262, 0, 0, 0x003BF, 0, 0, 0, 0, 0, 0, 0x02232, 0, 0x02275, 0, 0, 0,
    0, 0x02A70, 0, 0, 0, 0x020AC, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00392, 0x02281, 0, 0, 0, 0, 0, 0, 0,
    0x0227B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A37, 0, 0, 0, 0x00426, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x003C6, 0, 0, 0, 0x1D51E, 0, 0, 0x02AD4, 0, 0, 0x02A81, 0, 0x02257, 0, 0, 0, 0, 0, 0, 0x000AE,
    0, 0, 0, 0, 0x022DA, 0, 0, 0, 0, 0x000CD, 0x0042E, 0, 0, 0, 0, 0x0040C, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x000EB, 0, 0, 0, 0, 0, 0, 0, 0, 0x00102, 0, 0, 0, 0, 0, 0x00403, 0, 0x02AD6,
    0, 0, 0x000FC, 0x1D4C0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x027EB, 0, 0x0227D, 0, 0, 0, 0, 0, 0, 0,
    0x0290F, 0x000E8, 0x0011E, 0, 0, 0, 0, 0, 0, 0, 0x0222E, 0, 0, 0, 0, 0, 0, 0, 0x1D56A, 0x0039A,
    0, 0x000F9, 0, 0x02253, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02022,
    0x02A89, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02256, 0x000B0, 0, 0x02557, 0, 0x00172, 0, 0,
    0x021B1, 0x0227D, 0, 0, 0, 0, 0, 0, 0, 0, 0x02137, 0, 0, 0, 0, 0x0255D, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02550, 0x02A50, 0, 0, 0, 0x00409, 0, 0x02283, 0, 0, 0, 0x0003F, 0, 0, 0, 0, 0, 0, 0, 0,
    0x000CB, 0, 0, 0x0003E, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A8E, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x00407, 0, 0, 0, 0, 0, 0, 0x029B6, 0x023DC, 0, 0, 0, 0, 0x000C8, 0, 0, 0, 0, 0,
    0x029B3, 0, 0x02563, 0x0220B, 0, 0x029B4, 0, 0x022D7, 0, 0, 0x02041, 0x02230, 0, 0, 0, 0,
    0x0200B, 0x00101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00133, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0x000A5, 0x0040A, 0, 0x022D1, 0, 0, 0x000EF, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x022C2, 0, 0, 0, 0, 0, 0, 0, 0x02A7E, 0, 0, 0, 0, 0, 0, 0, 0x02A7A, 0, 0, 0, 0, 0, 0x02978,
    0x02292, 0, 0, 0x022E2, 0, 0, 0, 0x02208, 0, 0, 0x000EC, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02257, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022E3, 0, 0, 0, 0x022A3, 0, 0, 0, 0, 0, 0x022A5,
    0x0227F, 0x1D4CE, 0, 0x0000A, 0, 0, 0, 0, 0, 0, 0, 0x000C4, 0, 0x001F5, 0x02254, 0, 0, 0x021CB,
    0, 0, 0, 0, 0, 0, 0, 0x02A01, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x000D6, 0x000C0, 0x02280, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x02A40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02271, 0, 0, 0x00169, 0, 0, 0, 0, 0x02AA9,
    0x000D2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D552, 0, 0, 0, 0, 0x021BF, 0, 0, 0x02958, 0, 0, 0x029C5,
    0, 0, 0, 0, 0, 0, 0, 0, 0x022A3, 0, 0x02995, 0x00131, 0, 0, 0, 0, 0, 0, 0x02736, 0x0FB02,
    0x02190, 0, 0, 0, 0, 0, 0x02061, 0, 0x00178, 0, 0, 0, 0x02026, 0, 0, 0x02276, 0, 0x000F1, 0,
    0x002D8, 0, 0, 0, 0, 0, 0x02A9A, 0, 0x02A06, 0, 0, 0x02193, 0, 0x00026, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0296F, 0, 0, 0, 0, 0, 0, 0, 0, 0x0225C, 0, 0, 0x022C3, 0, 0, 0, 0, 0x021E4, 0, 0x021D2, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0x0010A, 0, 0, 0, 0x002DB, 0x02A6F, 0, 0, 0, 0x0FB01, 0x021D1, 0, 0, 0, 0,
    0, 0, 0, 0, 0x027ED, 0x022B4, 0, 0, 0, 0x027EC, 0, 0, 0, 0, 0, 0, 0, 0, 0x000A2, 0, 0, 0, 0,
    0x0040F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x002C6, 0, 0, 0x0002A, 0,
    0, 0x02299, 0, 0x02021, 0, 0x0228B, 0, 0, 0, 0, 0x02566, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02118, 0x00405, 0, 0, 0, 0x022EF, 0, 0x02A98, 0, 0, 0, 0, 0x02569, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x02022, 0, 0x00129, 0x02291, 0, 0x02008, 0, 0, 0, 0, 0x02A55, 0, 0, 0, 0, 0,
    0, 0x0228F, 0, 0, 0x0228F, 0, 0, 0x02204, 0, 0, 0, 0, 0, 0x025B8, 0, 0, 0x02124, 0x02AC6,
    0x025C2, 0, 0, 0x022D3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x1D4B6, 0,
    0x00157, 0, 0x022C8, 0, 0x025B4, 0x0013C, 0, 0, 0, 0, 0x0015F, 0, 0x022C0, 0, 0, 0, 0, 0, 0, 0,
    0x00163, 0, 0, 0x00446, 0, 0x0224F, 0x0297B, 0, 0, 0, 0x02A06, 0, 0x000C3, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x02605, 0, 0, 0, 0, 0, 0x022AC, 0, 0, 0, 0, 0x022AE, 0, 0x0015D, 0x00107, 0, 0, 0,
    0x000EA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000D5, 0, 0x027F5, 0, 0, 0, 0, 0, 0x000FB, 0, 0, 0,
    0, 0, 0, 0, 0x02213, 0x021D1, 0x02260, 0, 0, 0, 0, 0x00174, 0, 0x025BE, 0x00146, 0, 0, 0, 0, 0,
    0, 0, 0, 0x0227D, 0, 0, 0x02A58, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0015E, 0, 0x00134, 0, 0,
    0, 0, 0, 0x00455, 0, 0, 0, 0, 0, 0, 0x021C7, 0, 0x0016C, 0x02A49, 0, 0, 0, 0x0250C, 0, 0,
    0x0201D, 0x02102, 0x02910, 0, 0, 0x0201E, 0, 0, 0, 0x0040E, 0x022D9, 0, 0x02A84, 0x00427, 0, 0,
    0x0012E, 0, 0, 0, 0, 0, 0x0015C, 0x027E6, 0, 0, 0, 0x02233, 0, 0, 0, 0, 0, 0, 0x0220F, 0, 0, 0,
    0, 0x02252, 0, 0x0229A, 0, 0, 0, 0x021D5, 0, 0, 0, 0x02A11, 0x022AD, 0x02315, 0, 0, 0, 0x022AF,
    0, 0, 0x02A77, 0, 0, 0, 0x000CA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02287, 0x02AC6, 0, 0x00440, 0,
    0x003D1, 0, 0, 0x0043B, 0, 0, 0, 0, 0x00441, 0, 0, 0, 0, 0x0044D, 0, 0, 0, 0, 0x00442, 0x027E7,
    0x022EA, 0, 0, 0, 0x027E6, 0, 0, 0, 0, 0, 0x02297, 0, 0, 0x00443, 0, 0, 0, 0, 0x021D0, 0, 0,
    0x00399, 0, 0x00431, 0, 0, 0, 0, 0, 0, 0x022E9, 0, 0, 0x02AC8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0210B, 0, 0, 0, 0, 0, 0x02510, 0, 0, 0x0043F, 0, 0x000D1, 0, 0, 0x00419, 0x02019, 0, 0x0228B,
    0x02112, 0, 0x02018, 0, 0, 0, 0, 0, 0x000EE, 0, 0, 0, 0, 0, 0, 0, 0x0043D, 0, 0, 0x02226, 0,
    0x0230A, 0, 0, 0x02A7D, 0, 0x027F8, 0, 0, 0, 0, 0x027FA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00421, 0,
    0, 0x00022, 0, 0x00434, 0, 0x000F0, 0, 0x02013, 0, 0, 0, 0, 0, 0x00136, 0, 0, 0, 0, 0x00437, 0,
    0, 0, 0, 0x00417, 0, 0, 0, 0x02129, 0x0043C, 0, 0, 0, 0, 0, 0, 0, 0x00401, 0, 0x0042D, 0, 0, 0,
    0, 0x00122, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022A8, 0x02ACC, 0x000C2, 0, 0x02146, 0, 0, 0, 0,
    0x02014, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0045B, 0, 0, 0, 0x0229D, 0, 0, 0, 0, 0, 0x003B4, 0, 0, 0,
    0, 0x003B8, 0x00130, 0, 0x0011C, 0, 0, 0, 0, 0, 0, 0x000D4, 0x02191, 0, 0, 0, 0, 0, 0, 0, 0,
    0x022D4, 0, 0, 0, 0, 0, 0, 0, 0x02266, 0, 0, 0, 0x02A6D, 0x003D1, 0, 0, 0x02A78, 0, 0, 0x000DC,
    0, 0x022B3, 0, 0, 0, 0x022B2, 0, 0, 0, 0, 0, 0, 0, 0, 0x00402, 0, 0x00438, 0, 0x027FF, 0x02224,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0013B, 0x02AA6, 0, 0, 0, 0x022AA, 0x000D9, 0,
    0, 0, 0, 0, 0, 0x000B7, 0, 0x00432, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00176, 0, 0, 0x02AB0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02282, 0x022A2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x000B7, 0, 0, 0x02AA5, 0x025CA, 0, 0, 0, 0x00410, 0x027F8, 0x02010, 0, 0, 0x0041A,
    0x027F9, 0, 0, 0, 0, 0, 0, 0x000AF, 0, 0, 0x00237, 0x02979, 0, 0, 0x0223F, 0x02A5C, 0, 0, 0, 0,
    0, 0, 0, 0, 0x023DE, 0, 0, 0, 0, 0x00413, 0, 0, 0, 0, 0x000A0, 0, 0x0041E, 0, 0, 0x00145, 0, 0,
    0, 0, 0, 0, 0x023B4, 0, 0, 0x0224E, 0, 0, 0x025BF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x021B7, 0, 0, 0, 0, 0, 0x02935, 0, 0, 0, 0x0222E, 0, 0, 0, 0, 0, 0x02938, 0x022D0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x0211C, 0, 0, 0, 0, 0, 0, 0x021A9, 0, 0, 0x003B3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0x000F6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0042B, 0, 0, 0x022A4, 0, 0, 0, 0, 0, 0, 0,
    0x00453, 0, 0, 0, 0, 0, 0, 0, 0, 0x0041B, 0, 0x02ABF, 0, 0, 0, 0, 0, 0, 0, 0, 0x000F2, 0, 0, 0,
    0x021BD, 0, 0, 0x02956, 0, 0x00444, 0, 0, 0, 0, 0x00156, 0, 0, 0x022C1, 0, 0, 0x02773, 0x02136,
    0, 0, 0, 0x02772, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02913, 0, 0x00412, 0, 0,
    0x02278, 0, 0, 0, 0, 0, 0x0FB00, 0, 0x021B6, 0, 0x00121, 0, 0, 0, 0, 0, 0, 0, 0, 0x022A0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022A9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02951,
    0, 0, 0x0224F, 0, 0, 0, 0x00153, 0x0041D, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0x00311, 0, 0, 0, 0, 0x02190, 0x02281, 0x00168, 0, 0, 0x00411, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x00162, 0, 0, 0, 0, 0, 0, 0x02244, 0x02134, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0041C, 0, 0, 0, 0x021C9, 0x0041F, 0, 0, 0, 0, 0, 0, 0, 0, 0x02248, 0, 0x02147, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x02AC1, 0, 0, 0, 0, 0, 0, 0, 0, 0x02021, 0x0293D, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02AA4, 0, 0, 0, 0, 0, 0x0293C, 0, 0, 0, 0, 0x02A8B, 0, 0, 0, 0, 0x00060, 0, 0, 0, 0, 0, 0, 0,
    0x00420, 0, 0, 0, 0, 0, 0, 0x0224D, 0, 0, 0, 0, 0x0210C, 0, 0, 0, 0, 0, 0, 0, 0, 0x022E1, 0,
    0x02ABE, 0, 0, 0x003A9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02552, 0, 0, 0, 0x0225F, 0, 0, 0x022EC,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02AD8, 0, 0, 0, 0, 0, 0, 0, 0x02250, 0, 0, 0x0210D, 0, 0, 0, 0, 0,
    0, 0, 0, 0x00447, 0x022DE, 0, 0, 0, 0, 0x02AE4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022F6, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A7B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000F5,
    0, 0x02ACC, 0x000B4, 0, 0, 0, 0, 0x0002C, 0, 0x00424, 0, 0, 0x00422, 0x027F5, 0, 0, 0, 0, 0,
    0x02A04, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00060, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x0237C, 0, 0, 0, 0, 0, 0x02273, 0, 0x00135, 0, 0, 0, 0, 0x02111, 0, 0, 0, 0, 0, 0,
    0x0227B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0221F, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x022CB, 0, 0, 0x02AD5, 0, 0, 0x02205, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0010B, 0x02514, 0, 0, 0, 0, 0, 0, 0, 0x021D3, 0, 0, 0, 0, 0, 0, 0, 0, 0x0016F, 0, 0x0212D,
    0x000DB, 0x000CF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022A1, 0, 0, 0, 0x02713, 0, 0, 0, 0,
    0x02AD3, 0, 0, 0, 0, 0, 0, 0, 0, 0x0011F, 0x000A4, 0, 0, 0, 0, 0, 0, 0, 0, 0x02196, 0x000CC, 0,
    0, 0, 0, 0, 0, 0x0295E, 0x02A85, 0, 0, 0, 0, 0, 0x02A7E, 0, 0x00124, 0, 0x00398, 0, 0, 0, 0, 0,
    0x02A45, 0, 0, 0, 0, 0, 0x02245, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021BC, 0, 0x0040B, 0, 0,
    0x02555, 0, 0, 0x00439, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x000C7, 0x02937, 0, 0, 0, 0, 0x02936, 0, 0x000FD, 0, 0, 0x02147, 0x02518, 0x0222A, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0x02282, 0x003C3, 0, 0, 0, 0x0200B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0x00108, 0, 0, 0, 0x02233, 0, 0, 0, 0, 0, 0, 0, 0, 0x00423, 0x02277, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x02926, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A31, 0, 0, 0, 0, 0, 0x02197,
    0x02AB6, 0, 0, 0, 0, 0, 0x02221, 0, 0, 0, 0x0212B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000F4, 0, 0, 0,
    0, 0, 0, 0x000C6, 0, 0, 0, 0, 0, 0, 0, 0, 0x022C1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022D0,
    0x003A3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0256C, 0, 0x00152, 0x029B2, 0x021AA,
    0x025BD, 0, 0, 0, 0x021A9, 0, 0, 0, 0, 0, 0, 0, 0, 0x02291, 0, 0, 0x027F6, 0, 0x0256B, 0,
    0x02019, 0, 0, 0, 0, 0, 0, 0x02287, 0x022FD, 0, 0, 0, 0x02A16, 0, 0, 0, 0, 0, 0, 0, 0x000C5, 0,
    0, 0x021D2, 0, 0x022BE, 0, 0, 0, 0, 0x0201D, 0, 0, 0, 0, 0x0201C, 0, 0x02A82, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0x003B7, 0, 0, 0, 0, 0, 0, 0, 0x02279, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0x0201E, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02584, 0, 0, 0, 0x002DC, 0,
    0, 0, 0x02605, 0, 0, 0x00128, 0, 0, 0x0043E, 0, 0, 0, 0, 0x0299D, 0, 0, 0, 0, 0, 0, 0x02AAF,
    0x02926, 0, 0, 0x02580, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0229D, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0252C, 0x02A8A, 0, 0x0266E, 0x02283, 0, 0,
    0, 0, 0x02115, 0, 0, 0, 0, 0, 0x0224B, 0, 0, 0, 0, 0, 0, 0, 0x000A9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x00397, 0, 0, 0x000E1, 0x00132, 0, 0, 0, 0x02923, 0, 0, 0, 0, 0, 0, 0, 0,
    0x00175, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x025AA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00414, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0x02925, 0, 0, 0, 0, 0x002C7, 0x0226B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02193, 0, 0, 0, 0x021AD, 0x021C3, 0, 0, 0x02959, 0, 0, 0, 0, 0, 0, 0x0228A, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0x027E9, 0, 0, 0, 0, 0, 0, 0, 0, 0x00394, 0, 0, 0, 0, 0, 0x0251C, 0, 0, 0x022D7, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02534, 0, 0, 0x02A8F, 0, 0, 0, 0, 0x027F7, 0, 0, 0, 0x02924, 0,
    0, 0, 0, 0x02009, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02AC5, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x02010, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x029F4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02135, 0, 0, 0, 0, 0, 0x0005D,
    0, 0, 0, 0, 0x0005B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000CE, 0, 0, 0, 0,
    0, 0, 0, 0, 0x02A46, 0, 0, 0, 0x02524, 0x02216, 0, 0, 0x003C9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x022EB, 0, 0, 0, 0, 0, 0, 0, 0, 0x00445, 0, 0x000D8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02558, 0, 0,
    0, 0, 0, 0x0226B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02AB7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0045C, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02502, 0, 0, 0, 0, 0, 0, 0, 0, 0x02286,
    0x02AC5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00418, 0x02A17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00415, 0, 0, 0, 0, 0, 0,
    0x0200A, 0x02AC7, 0, 0, 0, 0, 0, 0x0224C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0230B, 0, 0, 0, 0x0228A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A4A, 0x025B9, 0,
    0, 0, 0x022A2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02247, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0295B, 0x0210F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x002D9, 0x029CF, 0, 0x002DD, 0, 0, 0, 0, 0x022B4,
    0, 0, 0, 0, 0x022B2, 0, 0, 0, 0x023B5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02240, 0, 0, 0, 0, 0x003C2,
    0, 0, 0, 0, 0, 0, 0x02ACB, 0, 0x02ADA, 0, 0, 0, 0, 0x029D0, 0, 0, 0, 0, 0x0044F, 0, 0x022B5, 0,
    0x021AD, 0, 0x0011D, 0x022B3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02961, 0, 0, 0, 0, 0x0200B, 0,
    0, 0, 0, 0, 0x0016E, 0, 0, 0, 0, 0x027FA, 0, 0, 0x0255B, 0, 0, 0, 0, 0, 0x022CE, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022DB, 0, 0,
    0, 0, 0, 0, 0x02AD7, 0, 0, 0x0224D, 0, 0, 0, 0x02207, 0, 0, 0, 0x025AB, 0, 0, 0, 0, 0x02145, 0,
    0, 0, 0, 0, 0, 0, 0x023B1, 0, 0, 0, 0, 0x023B0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x02A5A, 0, 0, 0, 0, 0x00457, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021C5, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02205, 0, 0, 0, 0, 0,
    0x00433, 0x0219A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00404, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0224E, 0, 0, 0, 0, 0,
    0x025B3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021CA, 0, 0, 0, 0, 0, 0x0266A, 0, 0, 0, 0,
    0x0042C, 0, 0, 0, 0, 0, 0x00105, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0295C, 0, 0, 0, 0, 0, 0x0FB04, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003BB,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003BA,
    0, 0, 0, 0x0255E, 0, 0, 0, 0, 0, 0x000E7, 0, 0, 0, 0x00456, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02308, 0, 0, 0, 0, 0, 0, 0,
    0x00109, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022ED, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0x02AA7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02249, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0201D, 0, 0, 0, 0x027F7, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0x02138, 0, 0, 0, 0, 0, 0x02ABD, 0, 0, 0, 0, 0x021DA, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0x00332, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02043, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000B4, 0, 0x02AC4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00125,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00142, 0, 0,
    0, 0, 0, 0x022B5, 0x02ACB, 0, 0, 0, 0, 0x02561, 0, 0, 0x00167, 0, 0, 0, 0x02220, 0x022C0, 0, 0,
    0, 0, 0, 0, 0x000F8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x0295F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00391, 0, 0, 0, 0, 0, 0, 0x021A4, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021CD, 0, 0, 0, 0x00111, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x021BB, 0, 0, 0, 0x021BA, 0, 0, 0x02199, 0, 0, 0, 0, 0, 0, 0x0223E, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x0039B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A8C, 0,
    0, 0, 0, 0x02202, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0x0224A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02565, 0, 0, 0, 0, 0,
    0, 0x02194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02568, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000FF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x02960, 0, 0, 0, 0, 0, 0, 0x02198, 0, 0, 0, 0, 0x021C0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x00103, 0, 0, 0x021A5, 0x02135, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021F5, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x025B3, 0, 0, 0, 0x02A7C, 0x021B3, 0, 0, 0, 0, 0x021B2, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x02220, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x025EF, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02286, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x025BD, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00141, 0, 0, 0, 0, 0, 0, 0, 0x02A38, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00454, 0,
    0, 0, 0x021CB, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021CC, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x0FB03, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02063, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A04, 0x027FC, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x000E4, 0, 0, 0x02967, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x0228D, 0, 0, 0, 0, 0, 0, 0, 0x00137, 0, 0, 0, 0, 0x000E0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022CF, 0, 0, 0, 0, 0, 0x02A86, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02969, 0, 0, 0x000E6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0042A, 0, 0, 0, 0, 0, 0,
    0, 0x02031, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022CC, 0, 0, 0, 0,
    0, 0, 0, 0, 0x0295A, 0, 0, 0, 0, 0x022C7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x00166, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02950, 0, 0,
    0, 0, 0x021C6, 0, 0, 0, 0, 0, 0, 0, 0, 0x0043A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02130, 0, 0, 0, 0, 0, 0, 0x025EF, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x02953, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0200B, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x029A4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0x00177, 0, 0, 0, 0x003F5, 0x000E3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021A7, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0219B, 0x022BA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021C2,
    0, 0, 0, 0x021C3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022DF,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021BC, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02925, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0044B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00408, 0, 0, 0, 0, 0, 0,
    0x00126, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x021BD, 0, 0, 0, 0, 0x000B8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021CF, 0, 0x02209, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02AB0, 0, 0, 0, 0, 0, 0, 0x00406, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0x000E2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x00429, 0, 0, 0, 0, 0, 0, 0x0210E, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02500, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0294E, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00110, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0210F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0x00430, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A36, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02248, 0, 0,
    0x029C9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x021DB, 0, 0, 0, 0, 0, 0, 0, 0x02195, 0, 0, 0, 0x0007C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00138, 0, 0, 0, 0, 0, 0, 0, 0, 0x0294F,
    0, 0x0220C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02AC3, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x027F6, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x027E7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x021C6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022D9, 0, 0, 0, 0, 0, 0, 0x021C4, 0, 0, 0, 0, 0,
    0x0256A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0253C, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02192, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021C4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x0295D, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0232D, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A29,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x00458, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021A6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x0299A, 0, 0, 0, 0, 0, 0, 0, 0x02AB8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0219D, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x027F9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021C1, 0, 0, 0x02957, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022B8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021AA, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x000E5, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022DE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A3F, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022CE, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02952, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x029AC, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x029A9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00449, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02911,
    0x02713, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02248, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02966, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02AB9, 0, 0, 0x029AB, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021D5,
    0x0044C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02968, 0, 0, 0, 0, 0, 0x00127, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02286, 0x0222B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x023DD, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003DD, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x029AD, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0044E, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0x00171, 0x022ED, 0, 0, 0, 0, 0x022EB, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0x027E8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x02194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x00150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021A0, 0, 0, 0, 0, 0, 0x021BE,
    0, 0, 0x02954, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x003D5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02287, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x022CF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x02148, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x003B1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021D4, 0, 0, 0, 0, 0, 0, 0,
    0x0044A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0x0294B, 0, 0, 0, 0, 0, 0, 0, 0x0294A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x00170, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02309, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0x021C2, 0, 0, 0x02955, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x00151, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021E5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x029AE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02222, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02060, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x022DF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02192, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0x029AA, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02A48, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x029AF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0x02223, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02758, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x02ABA, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021D2, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x0219E, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021C0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021C1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0x029A8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0x0220B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021A3,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021B6, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x021B7,
];
