pub mod offset {
    use crate::registers::id;
    use crate::LangError;
    use crate::LangError::InvalidRegisterId;

    pub const AH: usize = 0;
    pub const AL: usize = 1;
    pub const BH: usize = 2;
    pub const BL: usize = 3;
    pub const CH: usize = 4;
    pub const CL: usize = 5;
    pub const DH: usize = 6;
    pub const DL: usize = 7;
    pub const FLAGS: usize = 8;
    pub const AX: usize = AH;
    pub const BX: usize = BH;
    pub const CX: usize = CH;
    pub const DX: usize = DH;

    pub const fn from_id(id: u8) -> Result<u8, LangError> {
        let offset = match id as usize {
            id::AH | id::AX => AH,
            id::BH | id::BX => BH,
            id::CH | id::CX => CH,
            id::DH | id::DX => DH,
            id::AL => AL,
            id::BL => BL,
            id::CL => CL,
            id::DL => DL,
            id::FLAGS => FLAGS,
            _ => return Err(InvalidRegisterId(id)),
        };
        Ok(offset as u8)
    }
}

pub mod id {
    use crate::LangError;
    use crate::LangError::{InvalidRegisterId, InvalidRegisterName};

    pub const AH: usize = 0;
    pub const AL: usize = 1;
    pub const BH: usize = 2;
    pub const BL: usize = 3;
    pub const CH: usize = 4;
    pub const CL: usize = 5;
    pub const DH: usize = 6;
    pub const DL: usize = 7;
    pub const FLAGS: usize = 8;
    pub const AX: usize = 9;
    pub const BX: usize = 10;
    pub const CX: usize = 11;
    pub const DX: usize = 12;

    pub const fn size(value: usize) -> usize {
        if matches!(value, AX | BX | CX | DX) {
            2
        } else {
            1
        }
    }

    pub fn to_name(value: u8) -> Result<&'static str, LangError> {
        let name = match value as usize {
            AH => "AH",
            AL => "AL",
            BH => "BH",
            BL => "BL",
            CH => "CH",
            CL => "CL",
            DH => "DH",
            DL => "DL",
            AX => "AX",
            BX => "BX",
            CX => "CX",
            DX => "DX",
            FLAGS => "FLG",
            _ => return Err(InvalidRegisterId(value)),
        };
        Ok(name)
    }

    pub fn from_name(value: &str) -> Result<usize, LangError> {
        let code = match value.to_ascii_uppercase().as_str() {
            "AH" => AH,
            "AL" => AL,
            "BH" => BH,
            "BL" => BL,
            "CH" => CH,
            "CL" => CL,
            "DH" => DH,
            "DL" => DL,
            "AX" => AX,
            "BX" => BX,
            "CX" => CX,
            "DX" => DX,
            "FLG" => FLAGS,
            _ => return Err(InvalidRegisterName(value.to_string())),
        };
        Ok(code)
    }
}

pub const SIZE: usize = 9;

pub mod flags {
    pub const CARRY: u8 = 0b10000000;
    pub const ZERO: u8 = 0b01000000;
    pub const SIGNED: u8 = 0b00100000;
    pub const OVERFLOW: u8 = 0b00010000;
    pub const LESS_THAN: u8 = 0b00001000;
    pub const GREATER_THAN: u8 = 0b00000100;
    //pub const RESERVED: u8 = 0b00000010;
    pub const INTERRUPTS: u8 = 0b00000001;
}

pub const FLG_DEFAULT: u8 = flags::ZERO + flags::INTERRUPTS;

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn check_ids_align() {
        assert_eq!(offset::from_id(id::AH as u8).unwrap(), offset::AH as u8);
        assert_eq!(offset::from_id(id::AL as u8).unwrap(), offset::AL as u8);
        assert_eq!(offset::from_id(id::BH as u8).unwrap(), offset::BH as u8);
        assert_eq!(offset::from_id(id::BL as u8).unwrap(), offset::BL as u8);
        assert_eq!(offset::from_id(id::CH as u8).unwrap(), offset::CH as u8);
        assert_eq!(offset::from_id(id::CL as u8).unwrap(), offset::CL as u8);
        assert_eq!(offset::from_id(id::DH as u8).unwrap(), offset::DH as u8);
        assert_eq!(offset::from_id(id::DL as u8).unwrap(), offset::DL as u8);
        assert_eq!(offset::from_id(id::AX as u8).unwrap(), offset::AX as u8);
        assert_eq!(offset::from_id(id::BX as u8).unwrap(), offset::BX as u8);
        assert_eq!(offset::from_id(id::CX as u8).unwrap(), offset::CX as u8);
        assert_eq!(offset::from_id(id::DX as u8).unwrap(), offset::DX as u8);
        assert_eq!(
            offset::from_id(id::FLAGS as u8).unwrap(),
            offset::FLAGS as u8
        );

        assert_eq!(id::to_name(id::AH as u8).unwrap(), "AH");
        assert_eq!(id::to_name(id::AL as u8).unwrap(), "AL");
        assert_eq!(id::to_name(id::BH as u8).unwrap(), "BH");
        assert_eq!(id::to_name(id::BL as u8).unwrap(), "BL");
        assert_eq!(id::to_name(id::CH as u8).unwrap(), "CH");
        assert_eq!(id::to_name(id::CL as u8).unwrap(), "CL");
        assert_eq!(id::to_name(id::DH as u8).unwrap(), "DH");
        assert_eq!(id::to_name(id::DL as u8).unwrap(), "DL");
        assert_eq!(id::to_name(id::AX as u8).unwrap(), "AX");
        assert_eq!(id::to_name(id::BX as u8).unwrap(), "BX");
        assert_eq!(id::to_name(id::CX as u8).unwrap(), "CX");
        assert_eq!(id::to_name(id::DX as u8).unwrap(), "DX");
        assert_eq!(id::to_name(id::FLAGS as u8).unwrap(), "FLG");

        assert_eq!(id::from_name("AH").unwrap(), id::AH);
        assert_eq!(id::from_name("AL").unwrap(), id::AL);
        assert_eq!(id::from_name("BH").unwrap(), id::BH);
        assert_eq!(id::from_name("BL").unwrap(), id::BL);
        assert_eq!(id::from_name("CH").unwrap(), id::CH);
        assert_eq!(id::from_name("CL").unwrap(), id::CL);
        assert_eq!(id::from_name("DH").unwrap(), id::DH);
        assert_eq!(id::from_name("DL").unwrap(), id::DL);
        assert_eq!(id::from_name("ax").unwrap(), id::AX);
        assert_eq!(id::from_name("bX").unwrap(), id::BX);
        assert_eq!(id::from_name("Cx").unwrap(), id::CX);
        assert_eq!(id::from_name("DX").unwrap(), id::DX);
        assert_eq!(id::from_name("FLG").unwrap(), id::FLAGS);
    }

    #[test]
    fn check_invalid_values() {
        assert!(id::from_name("gsgad").is_err());
        assert!(id::to_name(255).is_err());
        assert!(offset::from_id(255).is_err());
    }
}
