use super::prelude::*;
use super::{uvec4, vec4, UVec4, Vec4};

#[cfg(target_arch = "spirv")]
use num_traits::Float;

/// Extensions to [`Vec4`]
///
/// Adds additional functionality to [`Vec4`] that [`glam`] doesn't have.
pub trait Vec4Ext {
    /// For element `i` of `self`, return `v[i].trunc()`
    fn trunc(self) -> Self;

    /// For element `i` of the return value, returns 0.0 if `value[i] < self[i]` and 1.0 otherwise.
    ///
    /// Similar to glsl's step(edge, x), which translates into edge.step(x)
    fn step(self, value: Self) -> Self;

    /// Selects between `true` and `false` based on the result of `value[i] < self[i]`
    fn step_select(self, value: Self, tru: Self, fals: Self) -> Self;

    /// Return only the fractional parts of each component.
    fn fract(self) -> Self;

    /// Return the square root of each component.
    fn sqrt(self) -> Self;

    /// Raw transmute each component to u32.
    fn to_bits(self) -> UVec4;
}

impl Vec4Ext for Vec4 {
    #[inline]
    fn trunc(self) -> Self {
        vec4(
            self.x.trunc(),
            self.y.trunc(),
            self.z.trunc(),
            self.w.trunc(),
        )
    }

    fn step(self, value: Vec4) -> Self {
        vec4(
            self.x.step(value.x),
            self.y.step(value.y),
            self.z.step(value.z),
            self.w.step(value.z),
        )
    }

    fn step_select(self, value: Self, less: Self, greater_or_equal: Self) -> Self {
        vec4(
            self.x.step_select(value.x, less.x, greater_or_equal.x),
            self.y.step_select(value.y, less.y, greater_or_equal.y),
            self.z.step_select(value.z, less.z, greater_or_equal.z),
            self.w.step_select(value.w, less.w, greater_or_equal.w),
        )
    }

    #[inline]
    fn fract(self) -> Self {
        vec4(
            self.x.fract(),
            self.y.fract(),
            self.z.fract(),
            self.w.fract(),
        )
    }

    fn sqrt(self) -> Self {
        vec4(self.x.sqrt(), self.y.sqrt(), self.z.sqrt(), self.w.sqrt())
    }

    fn to_bits(self) -> UVec4 {
        uvec4(
            self.x.to_bits(),
            self.y.to_bits(),
            self.z.to_bits(),
            self.w.to_bits(),
        )
    }
}
