use super::prelude::*;
use super::{vec3, Vec3};

#[cfg(target_arch = "spirv")]
use num_traits::Float;

/// Extensions to [`Vec3`]
///
/// Adds additional functionality to [`Vec3`] that [`glam`] doesn't have.
pub trait Vec3Ext {
    /// For element `i` of `self`, return `v[i].trunc()`
    fn trunc(self) -> Self;

    /// For element `i` of the return value, returns 0.0 if `value[i] < self[i]` and 1.0 otherwise.
    ///
    /// Similar to glsl's step(edge, x), which translates into edge.step(x)
    fn step(self, value: Self) -> Self;

    /// Selects between `true` and `false` based on the result of `value[i] < self[i]`
    fn step_select(self, value: Self, tru: Self, fals: Self) -> Self;

    /// Return only the fractional parts of each component.
    fn fract(self) -> Self;

    /// Clamp all components of `self` to the range `[0.0, 1.0]`
    fn saturate(self) -> Self;

    /// Square root of all three components.
    fn sqrt(self) -> Self;

    /// Natural logarithm of all three components.
    fn ln(self) -> Self;

    /// The reflection of a incident vector and surface normal.
    fn reflect(self, normal: Self) -> Self;
}

impl Vec3Ext for Vec3 {
    /// For element `i` of `self`, return `v[i].trunc()`
    #[inline]
    fn trunc(self) -> Self {
        vec3(self.x.trunc(), self.y.trunc(), self.z.trunc())
    }

    fn step(self, value: Vec3) -> Self {
        vec3(
            self.x.step(value.x),
            self.y.step(value.y),
            self.z.step(value.z),
        )
    }

    fn step_select(self, value: Self, less: Self, greater_or_equal: Self) -> Self {
        vec3(
            self.x.step_select(value.x, less.x, greater_or_equal.x),
            self.y.step_select(value.y, less.y, greater_or_equal.y),
            self.z.step_select(value.z, less.z, greater_or_equal.z),
        )
    }

    fn fract(self) -> Self {
        vec3(self.x.fract(), self.y.fract(), self.z.fract())
    }

    fn saturate(self) -> Self {
        vec3(self.x.saturate(), self.y.saturate(), self.z.saturate())
    }

    fn sqrt(self) -> Self {
        vec3(self.x.sqrt(), self.y.sqrt(), self.z.sqrt())
    }

    fn ln(self) -> Self {
        vec3(self.x.ln(), self.y.ln(), self.z.ln())
    }

    fn reflect(self, normal: Vec3) -> Self {
        self - 2.0 * normal * self.dot(normal)
    }
}

/// Coordinate system extension to [`Vec3`]
///
/// This crate is opinionated  with what coordinate system it uses and this adds
/// additional functions to acccess the coordinate system axis
///
/// The exact coordinate system we use is right-handed with +X = right, +Y = up, -Z = forward, +Z = back
pub trait CoordinateSystem {
    /// A unit length vector pointing in the canonical up direction.
    fn up() -> Self;

    /// A unit length vector pointing in the canonical down direction.
    fn down() -> Self;

    /// A unit length vector pointing in the canonical right direction.
    ///
    /// This is the right hand side of a first person character.
    fn right() -> Self;

    /// A unit length vector pointing in the canonical left direction.
    fn left() -> Self;

    /// A unit length vector pointing in the canonical forward direction.
    ///
    /// This is the direction a character faces, or a car drives towards.
    fn forward() -> Self;

    /// A unit length vector pointing in the canonical back direction.
    fn back() -> Self;
}

impl CoordinateSystem for Vec3 {
    fn up() -> Self {
        Self::new(0.0, 1.0, 0.0)
    }

    fn down() -> Self {
        Self::new(0.0, -1.0, 0.0)
    }

    fn right() -> Self {
        Self::new(1.0, 0.0, 0.0)
    }

    fn left() -> Self {
        Self::new(-1.0, 0.0, 0.0)
    }

    fn forward() -> Self {
        Self::new(0.0, 0.0, -1.0)
    }

    fn back() -> Self {
        Self::new(0.0, 0.0, 1.0)
    }
}
