use super::prelude::*;
use super::{vec2, Vec2};

#[cfg(target_arch = "spirv")]
use num_traits::Float;

/// Extensions to [`Vec2`]
///
/// Adds additional functionality to [`Vec2`] that [`glam`] doesn't have.
pub trait Vec2Ext {
    /// `x: cos(angle), y: sin(angle)` (in radians)
    fn from_angle(angle: f32) -> Self;

    /// Angle of the vector: `y.atan2(x)`
    fn angle(&self) -> f32;

    /// For element `i` of `self`, return `v[i].trunc()`
    fn trunc(self) -> Self;

    /// For element `i` of the return value, returns 0.0 if `value[i] < self[i]` and 1.0 otherwise.
    ///
    /// Similar to glsl's step(edge, x), which translates into edge.step(x)
    fn step(self, value: Self) -> Self;

    /// Selects between `true` and `false` based on the result of `value[i] < self[i]`
    fn step_select(self, value: Self, tru: Self, fals: Self) -> Self;

    /// Return only the fractional parts of each component.
    fn fract(self) -> Self;

    /// Clamp all components of `self` to the range `[0.0, 1.0]`
    fn saturate(self) -> Self;
}

impl Vec2Ext for Vec2 {
    #[inline]
    fn trunc(self) -> Self {
        vec2(self.x.trunc(), self.y.trunc())
    }

    fn from_angle(angle: f32) -> Self {
        Self::new(angle.cos(), angle.sin())
    }

    fn angle(&self) -> f32 {
        self.y.atan2(self.x)
    }

    fn step(self, value: Self) -> Self {
        vec2(self.x.step(value.x), self.y.step(value.y))
    }

    fn step_select(self, value: Self, less: Self, greater_or_equal: Self) -> Self {
        vec2(
            self.x.step_select(value.x, less.x, greater_or_equal.x),
            self.y.step_select(value.y, less.y, greater_or_equal.y),
        )
    }

    fn fract(self) -> Self {
        vec2(self.x.fract(), self.y.fract())
    }

    fn saturate(self) -> Self {
        vec2(self.x.saturate(), self.y.saturate())
    }
}
