use super::prelude::*;
use super::{mat3, BoundingBox, Mat3, Mat4};

/// Extensions to [`Mat4`]
///
/// Adds additional functionality to [`Mat4`] that [`glam`] doesn't have.
pub trait Mat4Ext {
    /// Truncates `self` by one in each dimension into a [`Mat3`]
    fn to_mat3(&self) -> Mat3;

    /// Create a right-handed orthographic projection matrix from a [`BoundingBox`]
    fn orthographic_rh_from_bounding_box(bbox: BoundingBox) -> Self;
}

impl Mat4Ext for Mat4 {
    fn to_mat3(&self) -> Mat3 {
        mat3(self.x_axis.xyz(), self.y_axis.xyz(), self.z_axis.xyz())
    }

    /// Assumes +x right, +y up, +z *out* of the screen (i.e. further away from the camera gets more negative).
    ///
    /// Maps X and Y into [-1.0, +1.0] and Z into 0.0 near to 1.0 far.
    fn orthographic_rh_from_bounding_box(bbox: BoundingBox) -> Self {
        Self::orthographic_rh(
            bbox.min.x,
            bbox.max.x,
            bbox.min.y,
            bbox.max.y,
            -bbox.max.z,
            -bbox.min.z,
        )
    }
}
