/// Extensions to floating-point primitives.
///
/// Adds additional math-related functionality to floats
pub trait FloatExt {
    /// Returns `0.0` if `self < edge` and 1.0 otherwise.
    fn step(self, edge: Self) -> Self;

    /// Selects between `less` and `greater_or_equal` based on the result of `self < edge`
    fn step_select(self, edge: Self, less: Self, greater_or_equal: Self) -> Self;

    /// Performs a linear interpolation between `self` and `other` using `a` to weight between them.
    /// The return value is computed as `self * (1−a) + other * a`.
    fn lerp(self, other: Self, a: Self) -> Self;

    /// Clamp `self` within the range `[0.0, 1.0]`
    fn saturate(self) -> Self;
}

impl FloatExt for f32 {
    fn step(self, edge: Self) -> Self {
        if self < edge {
            0.0
        } else {
            1.0
        }
    }

    fn step_select(self, edge: Self, less: Self, greater_or_equal: Self) -> Self {
        if self < edge {
            less
        } else {
            greater_or_equal
        }
    }

    #[allow(clippy::suboptimal_flops)]
    fn lerp(self, other: Self, a: Self) -> Self {
        self + (other - self) * a
    }

    fn saturate(self) -> Self {
        self.clamp(0.0, 1.0)
    }
}
