use super::prelude::*;
use super::{vec3, Vec3};

#[cfg(target_arch = "spirv")]
use num_traits::Float;

/// Extensions to [`Vec3`]
///
/// Adds additional functionality to [`Vec3`] that [`glam`] doesn't have.
pub trait Vec3Ext {
    /// For element `i` of the return value, returns 0.0 if `v[i] < edge[i]` and 1.0 otherwise.
    fn step(self, edge: Self) -> Self;

    /// Selects between `true` and `false` based on the result of `v[i] < edge[i]`
    fn step_select(self, edge: Self, tru: Self, fals: Self) -> Self;

    /// Return only the fractional parts of each component.
    fn fract(self) -> Self;

    /// Clamp all components of `self` to the range `[0.0, 1.0]`
    fn saturate(self) -> Self;

    /// Square root of all three components.
    fn sqrt(self) -> Self;

    /// Natural logarithm of all three components.
    fn ln(self) -> Self;
}

impl Vec3Ext for Vec3 {
    fn step(self, edge: Self) -> Self {
        vec3(
            self.x.step(edge.x),
            self.y.step(edge.y),
            self.z.step(edge.z),
        )
    }

    fn step_select(self, edge: Self, less: Self, greater_or_equal: Self) -> Self {
        vec3(
            self.x.step_select(edge.x, less.x, greater_or_equal.x),
            self.y.step_select(edge.y, less.y, greater_or_equal.y),
            self.z.step_select(edge.z, less.z, greater_or_equal.z),
        )
    }

    fn fract(self) -> Self {
        vec3(self.x.fract(), self.y.fract(), self.z.fract())
    }

    fn saturate(self) -> Self {
        vec3(self.x.saturate(), self.y.saturate(), self.z.saturate())
    }

    fn sqrt(self) -> Self {
        vec3(self.x.sqrt(), self.y.sqrt(), self.z.sqrt())
    }

    fn ln(self) -> Self {
        vec3(self.x.ln(), self.y.ln(), self.z.ln())
    }
}

/// Coordinate system extension to [`Vec3`]
///
/// This crate is opinionated  with what coordinate system it uses and this adds
/// additional functions to acccess the coordinate system axis
///
/// The exact coordinate system we use is right-handed with +X = right, +Y = up, -Z = forward, +Z = back
pub trait CoordinateSystem {
    /// A unit length vector pointing in the canonical up direction.
    fn up() -> Self;

    /// A unit length vector pointing in the canonical down direction.
    fn down() -> Self;

    /// A unit length vector pointing in the canonical right direction.
    ///
    /// This is the right hand side of a first person character.
    fn right() -> Self;

    /// A unit length vector pointing in the canonical left direction.
    fn left() -> Self;

    /// A unit length vector pointing in the canonical forward direction.
    ///
    /// This is the direction a character faces, or a car drives towards.
    fn forward() -> Self;

    /// A unit length vector pointing in the canonical back direction.
    fn back() -> Self;
}

impl CoordinateSystem for Vec3 {
    fn up() -> Self {
        Self::new(0.0, 1.0, 0.0)
    }

    fn down() -> Self {
        Self::new(0.0, -1.0, 0.0)
    }

    fn right() -> Self {
        Self::new(1.0, 0.0, 0.0)
    }

    fn left() -> Self {
        Self::new(-1.0, 0.0, 0.0)
    }

    fn forward() -> Self {
        Self::new(0.0, 0.0, -1.0)
    }

    fn back() -> Self {
        Self::new(0.0, 0.0, 1.0)
    }
}
