use super::Vec3;
#[cfg(not(target_arch = "spirv"))]
use super::{Mat4, Quat};

/// A 3-dimensional axis-aligned bounding box
#[derive(Clone, Copy, Default, PartialEq)]
#[cfg_attr(feature = "with_serde", derive(serde::Serialize, serde::Deserialize))]
pub struct BoundingBox {
    /// Bounding box minimum
    pub min: Vec3,
    /// Bounding box maximum
    pub max: Vec3,
}

#[cfg(not(target_arch = "spirv"))]
impl core::fmt::Debug for BoundingBox {
    fn fmt(&self, f: &mut core::fmt::Formatter<'_>) -> core::fmt::Result {
        write!(f, "{:?} - {:?}", self.min, self.max)
    }
}

#[allow(unused)]
impl BoundingBox {
    /// Create a bounding box from a minimum and maximum position
    pub fn from_min_max(min: Vec3, max: Vec3) -> Self {
        Self { min, max }
    }

    /// Create a bounding box from a center position and a size
    pub fn from_center_size(center: Vec3, size: Vec3) -> Self {
        Self::from_min_max(center - 0.5 * size, center + 0.5 * size)
    }

    /// Create a bounding box from an iterator of points that the bounding box will cover
    pub fn from_points(points: impl Iterator<Item = Vec3>) -> Self {
        let mut bb = Self::nothing();
        for p in points {
            bb.extend(p);
        }
        bb
    }

    pub fn nothing() -> Self {
        Self {
            min: Vec3::splat(core::f32::INFINITY),
            max: Vec3::splat(core::f32::NEG_INFINITY),
        }
    }

    pub fn everything() -> Self {
        Self {
            min: Vec3::splat(core::f32::NEG_INFINITY),
            max: Vec3::splat(core::f32::INFINITY),
        }
    }

    /// Returns the center point of the bounding box
    pub fn center(&self) -> Vec3 {
        (self.min + self.max) * 0.5
    }

    /// Returns the 3D axis size of the bounding box
    pub fn size(&self) -> Vec3 {
        self.max - self.min
    }

    /// Returns half the size, kind of the radius. Common operation so gets its own function.
    pub fn half_size(&self) -> Vec3 {
        0.5 * (self.max - self.min)
    }

    /// Only correct for positively sized boxes
    pub fn volume(&self) -> f32 {
        let s = self.size();
        s.x * s.y * s.z
    }

    /// True if and only if there is no point for which `bb.contains(point)` is true.
    /// The opposite of `is_something()`.
    pub fn is_nothing(&self) -> bool {
        self.max.x < self.min.x || self.max.y < self.min.y || self.max.z < self.min.z
    }

    /// True if and only if there is at least one point for which `bb.contains(point)` is true.
    /// The opposite of `is_nothing()`.
    pub fn is_something(&self) -> bool {
        self.min.x <= self.max.x && self.min.y <= self.max.y && self.min.z <= self.max.z
    }

    /// true if min == max
    pub fn is_point(&self) -> bool {
        self.min == self.max
    }

    /// Checks if the bounding box only contains finite numbers
    pub fn is_finite(&self) -> bool {
        self.min.is_finite() && self.max.is_finite()
    }

    /// The eight corners of this bounding box
    pub fn corners(&self) -> [Vec3; 8] {
        [
            Vec3::new(self.min.x, self.min.y, self.min.z),
            Vec3::new(self.min.x, self.min.y, self.max.z),
            Vec3::new(self.min.x, self.max.y, self.min.z),
            Vec3::new(self.min.x, self.max.y, self.max.z),
            Vec3::new(self.max.x, self.min.y, self.min.z),
            Vec3::new(self.max.x, self.min.y, self.max.z),
            Vec3::new(self.max.x, self.max.y, self.min.z),
            Vec3::new(self.max.x, self.max.y, self.max.z),
        ]
    }

    pub fn extend(&mut self, pos: Vec3) {
        self.min = self.min.min(pos);
        self.max = self.max.max(pos);
    }

    #[must_use]
    pub fn union(mut self, other: Self) -> Self {
        self.min = self.min.min(other.min);
        self.max = self.max.max(other.max);
        self
    }

    #[must_use]
    pub fn intersection(mut self, other: Self) -> Self {
        let intersection = Self {
            min: self.min.max(other.min),
            max: self.max.max(other.max),
        };
        if intersection.is_nothing() {
            Self::nothing()
        } else {
            intersection
        }
    }

    /// Returns `true` if the point is within or on the edge of the box.
    #[must_use]
    pub fn contains(&self, point: Vec3) -> bool {
        (self.min.x <= point.x && point.x <= self.max.x)
            && (self.min.y <= point.y && point.y <= self.max.y)
            && (self.min.z <= point.z && point.z <= self.max.z)
    }

    /// Expand with this much padding on each side
    #[must_use]
    pub fn expanded(&self, padding: Vec3) -> Self {
        Self {
            min: self.min - padding,
            max: self.max + padding,
        }
    }

    #[must_use]
    pub fn translated(&self, translation: Vec3) -> Self {
        Self {
            min: self.min + translation,
            max: self.max + translation,
        }
    }

    #[must_use]
    #[cfg(not(target_arch = "spirv"))]
    pub fn rotated_around_origin(&self, q: &Quat) -> Self {
        if self.is_nothing() {
            Self::nothing()
        } else {
            // This can be optimized
            Self::from_points(self.corners().iter().map(|&c| q.mul_vec3(c)))
        }
    }

    #[must_use]
    #[cfg(not(target_arch = "spirv"))]
    pub fn transform(&self, m: &Mat4) -> Self {
        if self.is_nothing() {
            Self::nothing()
        } else {
            Self::from_points(self.corners().iter().map(|&c| m.transform_point3(c)))
        }
    }
}

#[test]
fn test_bounding_box() {
    let bb = BoundingBox::from_min_max(Vec3::ZERO, Vec3::ZERO);
    assert!(bb.contains(Vec3::ZERO));
    assert!(bb.is_something());
    assert!(!bb.is_nothing());
    let bb_rotated = bb.transform(&Mat4::from_quat(Quat::from_axis_angle(Vec3::X, 0.5)));
    assert_eq!(bb, bb_rotated);
    let bb_translated = bb.transform(&Mat4::from_translation(Vec3::new(2.0, 3.0, 5.0)));
    assert!(bb_translated.is_something());
    assert!(bb_translated.contains(Vec3::new(2.0, 3.0, 5.0)));
}
