use super::prelude::*;
use super::{vec2, Vec2};

#[cfg(target_arch = "spirv")]
use spirv_std::num_traits::Float;

/// Extensions to [`Vec2`]
///
/// Adds additional functionality to [`Vec2`] that [`glam`] doesn't have.
pub trait Vec2Ext {
    /// For element `i` of the return value, returns 0.0 if `v[i] < edge[i]` and 1.0 otherwise.
    fn step(self, edge: Self) -> Self;

    /// Selects between `true` and `false` based on the result of `v[i] < edge[i]`
    fn step_select(self, edge: Self, tru: Self, fals: Self) -> Self;

    /// Return only the fractional parts of each component.
    fn fract(self) -> Self;

    /// Clamp all components of `self` to the range `[0.0, 1.0]`
    fn saturate(self) -> Self;
}

impl Vec2Ext for Vec2 {
    fn step(self, edge: Self) -> Self {
        vec2(self.x.step(edge.x), self.y.step(edge.y))
    }

    fn step_select(self, edge: Self, less: Self, greater_or_equal: Self) -> Self {
        vec2(
            self.x.step_select(edge.x, less.x, greater_or_equal.x),
            self.y.step_select(edge.y, less.y, greater_or_equal.y),
        )
    }

    fn fract(self) -> Self {
        vec2(self.x.fract(), self.y.fract())
    }

    fn saturate(self) -> Self {
        vec2(self.x.saturate(), self.y.saturate())
    }
}
