use std::{
    convert::TryInto,
    marker::PhantomData,
    mem::{forget, transmute},
};

use crate::{
    environment::{params_to_vec, Param},
    error::LunaticError,
    host_api::{self, message, process},
    mailbox::{LinkMailbox, Mailbox, TransformMailbox},
    message::Message,
};

/// A sandboxed computation.
///
/// Processes are fundamental building blocks of Lunatic applications. Each of them has their own
/// memory space. The only way for processes to interact is trough [`Message`] passing.
///
/// ### Safety:
/// It's not safe to use mutable `static` variables to share data between processes, because each
/// of them is going to see a separate heap and a unique `static` variable.
pub struct Process<T: Message> {
    id: u64,
    _phantom: PhantomData<T>,
}

impl<T: Message> Clone for Process<T> {
    fn clone(&self) -> Self {
        let id = unsafe { host_api::process::clone_process(self.id) };
        Process::from(id)
    }
}

impl<T: Message> Drop for Process<T> {
    fn drop(&mut self) {
        unsafe { process::drop_process(self.id) };
    }
}

impl<T: Message> Message for Process<T> {
    fn from_bincode(data: &[u8], resources: &[u64]) -> (usize, Self) {
        // The serialized value for a process is the u64 index inside the resources array.
        // The resources array will contain the new resource index.
        let index = u64::from_le_bytes(data.try_into().unwrap());
        let proc = Process::from(resources[index as usize]);
        (8, proc)
    }

    #[allow(clippy::wrong_self_convention)]
    unsafe fn to_bincode(self, dest: &mut Vec<u8>) {
        let index = message::add_process(self.id);
        dest.extend(index.to_le_bytes());
        // By adding the process to the message it will be removed from our resources.
        // Dropping it would cause a trap.
        forget(self);
    }
}

impl<T: Message> Process<T> {
    pub(crate) fn from(id: u64) -> Self {
        Process {
            id,
            _phantom: PhantomData,
        }
    }

    pub fn send(&self, value: T) {
        // Create new message
        unsafe { message::create() };
        let mut buffer = Vec::new();
        unsafe { value.to_bincode(&mut buffer) };
        // Set sending buffer
        unsafe { message::set_buffer(buffer.as_ptr(), buffer.len()) };
        // During bincode serialization resources will add themself to the message
        // Send it
        unsafe { message::send(self.id) };
    }

    #[allow(clippy::result_unit_err)]
    pub fn join(&self) -> Result<(), ()> {
        match unsafe { process::join(self.id) } {
            0 => Ok(()),
            _ => Err(()),
        }
    }
}

/// Returns a handle to the current process.
pub fn this<T: Message, U: TransformMailbox<T>>(mailbox: U) -> (Process<T>, U) {
    let id = unsafe { process::this() };
    (Process::from(id), mailbox)
}

/// Spawns a new process from a function.
///
/// - `function` is the starting point of the new process. The new process doesn't share
///   memory with its parent, because of this the function can't capture anything from parents.
pub fn spawn<T: Message>(function: fn(Mailbox<T>)) -> Result<Process<T>, LunaticError> {
    // LinkMailbox<T> & Mailbox<T> are marker types and it's safe to cast to Mailbox<T> here if we
    //  set the `link` argument to `false`.
    let function = unsafe { transmute(function) };
    spawn_(None, false, Context::<(), _>::Without(function))
}

/// Spawns a new process from a function and links it to the parent.
///
/// - `function` is the starting point of the new process. The new process doesn't share
///   memory with its parent, because of this the function can't capture anything from parents.
pub fn spawn_link<T, P, M>(
    mailbox: M,
    function: fn(Mailbox<T>),
) -> Result<(Process<T>, LinkMailbox<P>), LunaticError>
where
    T: Message,
    P: Message,
    M: TransformMailbox<P>,
{
    let mailbox = mailbox.catch_child_panic();
    let proc = spawn_(None, true, Context::<(), _>::Without(function))?;
    Ok((proc, mailbox))
}

/// Spawns a new process from a function and links it to the parent.
///
/// - `function` is the starting point of the new process. The new process doesn't share
///   memory with its parent, because of this the function can't capture anything from parents.
///
/// If the linked process dies, the parent is going to die too.
pub fn spawn_link_unwrap<T, P, M>(
    mailbox: M,
    function: fn(Mailbox<T>),
) -> Result<(Process<T>, Mailbox<P>), LunaticError>
where
    T: Message,
    P: Message,
    M: TransformMailbox<P>,
{
    let mailbox = mailbox.panic_if_child_panics();
    let proc = spawn_(None, true, Context::<(), _>::Without(function))?;
    Ok((proc, mailbox))
}

/// Spawns a new process from a function and context.
///
/// - `context` is  data that we want to pass to the newly spawned process. It needs to impl.
///    the [`Message`] trait.
///
/// - `function` is the starting point of the new process. The new process doesn't share
///   memory with its parent, because of this the function can't capture anything from parents.
///   The first argument of this function is going to be the received `context`.
pub fn spawn_with<C: Message, T: Message>(
    context: C,
    function: fn(C, Mailbox<T>),
) -> Result<Process<T>, LunaticError> {
    // LinkMailbox<T> & Mailbox<T> are marker types and it's safe to cast to Mailbox<T> here if we
    //  set the `link` argument to `false`.
    let function = unsafe { transmute(function) };
    spawn_(None, false, Context::With(function, context))
}

/// Spawns a new process from a function and context, and links it to the parent.
///
/// - `context` is  data that we want to pass to the newly spawned process. It needs to impl.
///    the [`Message`] trait.
///
/// - `function` is the starting point of the new process. The new process doesn't share
///   memory with its parent, because of this the function can't capture anything from parents.
///   The first argument of this function is going to be the received `context`.
pub fn spawn_link_with<C, T, P, M>(
    mailbox: M,
    context: C,
    function: fn(C, Mailbox<T>),
) -> Result<(Process<T>, LinkMailbox<P>), LunaticError>
where
    C: Message,
    T: Message,
    P: Message,
    M: TransformMailbox<P>,
{
    let mailbox = mailbox.catch_child_panic();
    let proc = spawn_(None, true, Context::With(function, context))?;
    Ok((proc, mailbox))
}

/// Spawns a new process from a function and context, and links it to the parent.
///
/// - `context` is  data that we want to pass to the newly spawned process. It needs to impl.
///    the [`Message`] trait.
///
/// - `function` is the starting point of the new process. The new process doesn't share
///   memory with its parent, because of this the function can't capture anything from parents.
///   The first argument of this function is going to be the received `context`.
///
/// If the linked process dies, the parent is going to die too.
pub fn spawn_link_unwrap_with<C, T, P, M>(
    mailbox: M,
    context: C,
    function: fn(C, Mailbox<T>),
) -> Result<(Process<T>, Mailbox<P>), LunaticError>
where
    C: Message,
    T: Message,
    P: Message,
    M: TransformMailbox<P>,
{
    let mailbox = mailbox.panic_if_child_panics();
    let proc = spawn_(None, true, Context::With(function, context))?;
    Ok((proc, mailbox))
}

pub(crate) enum Context<C: Message, T: Message> {
    With(fn(C, Mailbox<T>), C),
    Without(fn(Mailbox<T>)),
}

// If `module_id` is None it will use the current module & environment.
pub(crate) fn spawn_<C: Message, T: Message>(
    module_id: Option<u64>,
    link: bool,
    context: Context<C, T>,
) -> Result<Process<T>, LunaticError> {
    // Spawning a new process from  the same module is a delicate undertaking.
    // First of all, the WebAssembly spec only allows us to call exported functions from a module
    // Therefore we define a module export under the name `_lunatic_spawn_by_index`. This global
    // function will get 2 arguments:
    // * A type helper function: `type_helper_wrapper_*`
    // * The function we want to use as an entry point: `function`
    // It's obvious why we need the entry function, but what is a type helper function? The entry
    // entry function contains 2 generic types, one for the context and one for messages, but the
    // `_lunatic_spawn_by_index` one can't be generic. That's why we use the type helper, to let
    // us wrap the call to the entry function into the right type signature.

    let (type_helper, func) = match context {
        Context::With(func, _) => (type_helper_wrapper_context::<C, T> as usize, func as usize),
        Context::Without(func) => (type_helper_wrapper::<T> as usize, func as usize),
    };
    let params = params_to_vec(&[Param::I32(type_helper as i32), Param::I32(func as i32)]);
    let mut id = 0;
    let func = "_lunatic_spawn_by_index";
    let result = unsafe {
        match module_id {
            Some(module_id) => host_api::process::spawn(
                if link { 1 } else { 0 },
                module_id,
                func.as_ptr(),
                func.len(),
                params.as_ptr(),
                params.len(),
                &mut id,
            ),
            None => host_api::process::inherit_spawn(
                if link { 1 } else { 0 },
                func.as_ptr(),
                func.len(),
                params.as_ptr(),
                params.len(),
                &mut id,
            ),
        }
    };
    if result == 0 {
        match context {
            // If context exists, send it as first message to the new process
            Context::With(_, context) => {
                let self_ = Process {
                    id,
                    _phantom: PhantomData,
                };
                self_.send(context);
                // Processes can only receive one type of messages, but to pass in the context we pretend
                // for the first message that our process is receiving messages of type `C`.
                Ok(unsafe { transmute(self_) })
            }
            Context::Without(_) => Ok(Process {
                id,
                _phantom: PhantomData,
            }),
        }
    } else {
        Err(LunaticError::from(id))
    }
}

/// Suspends the current process for `milliseconds`.
pub fn sleep(milliseconds: u64) {
    unsafe { host_api::process::sleep_ms(milliseconds) };
}

// Type helper
fn type_helper_wrapper<T: Message>(function: usize) {
    let mailbox = Mailbox::new();
    let function: fn(Mailbox<T>) = unsafe { transmute(function) };
    function(mailbox);
}

// Type helper with context
fn type_helper_wrapper_context<C: Message, T: Message>(function: usize) {
    let context = Mailbox::new().receive();
    let mailbox = Mailbox::new();
    let function: fn(C, Mailbox<T>) = unsafe { transmute(function) };
    function(context, mailbox);
}

#[export_name = "_lunatic_spawn_by_index"]
extern "C" fn _lunatic_spawn_by_index(type_helper: usize, function: usize) {
    let type_helper: fn(usize) = unsafe { transmute(type_helper) };
    type_helper(function);
}
