# 🦀 lunaria-api

[![GitHub release (latest by date)](https://img.shields.io/github/v/release/playlunaria/lunaria-api)](https://github.com/playlunaria/lunaria-api/releases)
[![GitHub Workflow Status](https://img.shields.io/github/workflow/status/playlunaria/lunaria-api/rust)](https://github.com/playlunaria/lunaria-api/actions?query=workflow%3ARust)
[![License](https://img.shields.io/crates/l/lunaria-api)](https://crates.io/crates/lunaria-api)

_A Rust API client for the video game Lunaria._

[Lunaria] is a video game for programmers, and is played by writing code that
interacts with the game through a [gRPC] API. This crate contains a [gRPC]
client that is auto-generated from the [Protocol Buffers][protobuf] that declare
[Lunaria's API][lunaria-api].

## Getting Started

First, add `lunaria-api` as a dependency to your `Cargo.toml`.

Because `lunaria-api` wraps a client generated by [`tonic`][tonic], it must be
added as a dependency as well. And if you are building a binary, you also need
an async runtime like [`tokio`][tokio].

```toml
[dependencies]
lunaria-api = "0.1.2"
tokio = { version = "0.2.22", features = ["macros", "rt-threaded"] }
tonic = "0.3.1"
```

Next, import `LunariaClient` and connect to the game server. Check out Lunaria's
API specification below to learn about all the requests you can send, and the
data they require and return:

<https://github.com/playlunaria/lunaria-api/tree/main/protobufs>

Here is an example that fetches the version of the game:

```rust
use lunaria_api::lunaria::v1::lunaria_service_client::LunariaSerrviceClient;
use lunaria_api::lunaria::v1::{GetVersionRequest, GetVersionResponse, Version};
use tonic::Request;

#[tokio::main]
async fn main() -> Result<(), Box<dyn std::error::Error>> {
    // Specify the address and port of Lunaria's API
    let address = "http://127.0.0.1:1904";

    // Initialize the client
    let mut lunaria = LunariaServiceClient::connect(address).await?;

    // Create a request to get the game's version and send it to the server
    let request = Request::new(GetVersionRequest {});
    let grpc_response = lunaria.get_version(request).await?;
    let version_response = grpc_response.into_inner();

    if let Some(version) = version_response.version {
        assert_eq!(0, version.major);
        assert_eq!(0, version.minor);
        assert_eq!(0, version.patch);
    }

    Ok(())
}
```

## License

Licensed under either of

- Apache License, Version 2.0 ([LICENSE-APACHE](LICENSE-APACHE) or <http://www.apache.org/licenses/LICENSE-2.0>)
- MIT license ([LICENSE-MIT](LICENSE-MIT) or <http://opensource.org/licenses/MIT>)

at your option.

## Contribution

Unless you explicitly state otherwise, any contribution intentionally submitted
for inclusion in the work by you, as defined in the Apache-2.0 license, shall be
dual licensed as above, without any additional terms or conditions.

[grpc]: https://grpc.io
[lunaria]: https://playlunaria.com
[lunaria-api]: https://github.com/playlunaria/lunaria-api
[protobuf]: https://developers.google.com/protocol-buffers/
[tokio]: https://github.com/tokio-rs/tokio
[tonic]: https://github.com/hyperium/tonic
