use luka::{Graph, algorithms, utils, Vertex};
use luka::algorithms::{dfs_with_visitor, VisitorDFS, VisitorDFSAction};
use luka::error::GraphError;

#[test]
fn integration_test_dfs_algorithm(){
    let mut graph = Graph::new(10);

    graph.add_edge(1, 2, 0.0).unwrap();
    graph.add_edge(2, 3, 0.0).unwrap();
    graph.add_edge(3, 5, 0.0).unwrap();

    let start = graph.get_vertex(1).unwrap();
    let target = graph.get_vertex(5).unwrap();
    let parents = algorithms::dfs(&graph, &start).unwrap();
    match utils::find_path(&graph, &target, &parents).unwrap() {
        Some(path) => {
            let mut ids = vec![];
            for vertex in path {
                ids.push(vertex.id());
            }
            assert_eq!(ids, vec![1, 2, 3, 5]);
        }
        None => {
            println!("Path not found !!!")
        }
    }
}

#[test]
#[should_panic]
fn integration_test_dfs_algorithm_panic() {
    let mut graph = Graph::new(100);
    graph.add_edge(1, 2, 0.0).unwrap();
    let start = graph.get_vertex(122).unwrap();
    algorithms::dfs(&graph, &start).unwrap();
}

#[test]
fn integration_test_dfs_with_visitor_lgorithm(){
    struct CustomVisitor {
        visiting_order: Vec<usize>
    }

    impl <T> VisitorDFS<T> for CustomVisitor {
        fn entry_to_vertex_event(&mut self, vertex: &Vertex<T>) -> Result<VisitorDFSAction, GraphError> {
            self.visiting_order.push(vertex.id());
            Ok(VisitorDFSAction::Nothing)
        }
    }

    let mut graph = Graph::new(10);

    graph.add_edge(1, 2, 0.0).unwrap();
    graph.add_edge(2, 3, 0.0).unwrap();
    graph.add_edge(3, 5, 0.0).unwrap();

    let mut visitor = CustomVisitor{ visiting_order: vec![] };
    let start = graph.get_vertex(1).unwrap();
    dfs_with_visitor(&graph, &start, &mut visitor).unwrap();

    assert_eq!(vec![1, 2, 3, 5], visitor.visiting_order);
}