use luka::{Graph, utils, Vertex};
use luka::algorithms;
use luka::algorithms::{VisitorBFS, VisitorBFSAction};
use luka::error::GraphError;

#[test]
fn integration_test_bfs_algorithm() {
    use crate::utils;

    let mut graph = Graph::new(100);
    graph.add_edge(1, 2, 0.0).unwrap();
    graph.add_edge(2, 3, 0.0).unwrap();
    graph.add_edge(2, 4, 0.0).unwrap();
    graph.add_edge(2, 5, 0.0).unwrap();
    graph.add_edge(4, 8, 0.0).unwrap();
    graph.add_edge(8, 17, 0.0).unwrap();

    let start = graph.get_vertex(1).unwrap();
    let target = graph.get_vertex(5).unwrap();
    let parents = algorithms::bfs(&graph, &start).unwrap();

    let path = utils::find_path(&graph, &target, &parents).unwrap().unwrap();
    let mut ids = vec![];
    for vertex in path {
        ids.push(vertex.id());
    }
    assert_eq!(ids, vec![1, 2, 5]);

    let target = graph.get_vertex(17).unwrap();
    let path = utils::find_path(&graph, &target, &parents).unwrap().unwrap();
    let mut ids = vec![];
    for vertex in path {
        ids.push(vertex.id());
    }
    assert_eq!(ids, vec![1, 2, 4, 8, 17]);
}

#[test]
#[should_panic]
fn integration_test_bfs_algorithm_panic() {
    let mut graph = Graph::new(100);
    graph.add_edge(1, 2, 0.0).unwrap();
    let start = graph.get_vertex(122).unwrap();
    algorithms::bfs(&graph, &start).unwrap();
}

#[test]
fn integration_test_bfs_with_visitor_algorithm() {
    struct CustomVisitor {
        visiting_order: Vec<usize>
    }

    impl <T> VisitorBFS<T> for CustomVisitor {
        fn queue_extract_event(&mut self, vertex: &Vertex<T>) -> Result<VisitorBFSAction, GraphError> {
            self.visiting_order.push(vertex.id());
            Ok(VisitorBFSAction::Nothing)
        }
    }

    let mut graph = Graph::new(100);
    graph.add_edge(1, 2, 0.0).unwrap();
    graph.add_edge(2, 3, 0.0).unwrap();
    graph.add_edge(2, 4, 0.0).unwrap();
    graph.add_edge(2, 5, 0.0).unwrap();
    graph.add_edge(4, 8, 0.0).unwrap();
    graph.add_edge(8, 17, 0.0).unwrap();

    let mut visitor = CustomVisitor{ visiting_order: vec![] };
    let start = graph.get_vertex(1).unwrap();
    algorithms::bfs_with_visitor(&graph, &start, &mut visitor).unwrap();

    assert_eq!(vec![1, 2, 3, 4, 5, 8, 17], visitor.visiting_order);
}